import deepEqual from 'deep-equal';
import {momentTimezone as moment} from '../../../reexports';

import {ROBOT} from '../date/formats';
import {BY} from '../countries';

import {getIncrementalTimeoutPromise} from '../timeoutPromise';
import {getStationFrom, getStationTo} from '../thread/thread';
import {getStationPointById} from '../point/pointType';
import noop from '../noop';

const isMobile = process.env.PLATFORM === 'mobile';

const timeout = getIncrementalTimeoutPromise();

export default class ThreadBlablacarPolling {
    constructor({store, api, callback = noop}) {
        this.api = api;
        this.callback = callback;
        this.context = this.getContext(store);
        this.timeStart = null;
        this.blablacar = {};
        this.allDaysPolling = false;
        this.shouldRestartPolling = false;
        this.stopped = false;
        this.done = false;
    }

    getContext(store) {
        const {thread, flags, nationalVersion, currencies} = store.getState();
        const {stations} = thread;

        const stationFromData = getStationFrom({stations});
        const stationToData = getStationTo({
            stations,
            stationFrom: stationFromData,
        });
        const {
            id: stationFromId,
            country: countryFrom,
            departureLocalDt: departureFrom,
        } = stationFromData || {};
        const {id: stationToId, country: countryTo} = stationToData || {};

        if (!stationFromId || !departureFrom || !stationToId) {
            return null;
        }

        const departureFromMoment = moment.parseZone(departureFrom);

        return {
            pointFrom: getStationPointById(stationFromId),
            pointTo: getStationPointById(stationToId),
            countryFrom,
            countryTo,
            departureFromMoment,
            currencies,
            flags,
            nationalVersion,
        };
    }

    requestPrices({poll = false, req}) {
        const {
            pointFrom,
            pointTo,
            departureFromMoment,
            nationalVersion,
            flags,
        } = this.context;

        if (this.needRequestBlablacar()) {
            return this.api.exec(
                'blablacar3',
                {
                    pointFrom,
                    pointTo,
                    ...(this.allDaysPolling
                        ? null
                        : {date: departureFromMoment.format(ROBOT)}),
                    nationalVersion,
                    poll,
                    isMobile,
                    flags,
                },
                req,
            );
        }

        return Promise.resolve();
    }

    /**
     * Проверяет эквивалентность контекста, с которым был создан поллинг, и переданного
     * @param {Object} store
     * @return {boolean}
     */
    isEqualContext(store) {
        return deepEqual(this.context, this.getContext(store));
    }

    /**
     * Проверяет актуальность полинга по времени
     * @return {boolean}
     */
    isActual() {
        return !this.timeStart || this.timeStart > moment().add(-5, 'minutes');
    }

    needRequestBlablacar() {
        const {
            pointFrom,
            pointTo,
            countryFrom,
            countryTo,
            departureFromMoment,
        } = this.context;
        const querying = this.blablacar.querying;

        return Boolean(
            !departureFromMoment.isBefore(moment(), 'day') &&
                pointFrom &&
                pointTo &&
                !(
                    countryFrom.code.toUpperCase() === BY &&
                    countryTo.code.toUpperCase() === BY
                ) &&
                (typeof querying === 'undefined' || querying === true),
        );
    }

    needRequestPrices() {
        if (this.stopped || this.done) {
            return false;
        }

        return this.needRequestBlablacar();
    }

    _setPriceAnswer(answer) {
        const {blablacar} = answer;
        let noOffersOnDate = false;
        let blablacarQuerying = false;

        if (blablacar) {
            noOffersOnDate =
                this.blablacar.querying !== false &&
                blablacar.querying === false &&
                Boolean(blablacar.tariff) &&
                !blablacar.tariff.offersCount &&
                !this.allDaysPolling;
            /**
             * Запрос к блаблакару активен если:
             * - в ответе на дату нет предпложений и нам нужно опросить ббк без даты
             * - так пришло в ответе
             */
            blablacarQuerying = noOffersOnDate || blablacar.querying;
        }

        this.blablacar = {
            ...this.blablacar,
            ...(this.allDaysPolling
                ? {
                      allDaysCheckComplete: blablacar.querying === false,
                      allDaysCheckResult: blablacar.tariff,
                  }
                : {
                      ...blablacar,
                      allDaysCheckComplete: false,
                      allDaysCheckResult: null,
                  }),
            querying: blablacarQuerying,
        };

        // Если на дату нет предложений, начинаем запрос ббк заново на все дни
        if (noOffersOnDate) {
            this.allDaysPolling = true;
            this.shouldRestartPolling = true;
        }
    }

    getPrices(poll = false, numberRetry = 0) {
        if (!this.needRequestPrices()) {
            return Promise.resolve(this.blablacar);
        }

        numberRetry++;

        return this.requestPrices({poll}).then(answer => {
            this._setPriceAnswer(answer);

            if (this.needRequestPrices()) {
                if (this.shouldRestartPolling) {
                    poll = false;
                    this.shouldRestartPolling = false;
                    numberRetry = 0;
                }

                return timeout(numberRetry).then(() =>
                    this.getPrices(poll, numberRetry),
                );
            }

            return this.blablacar;
        });
    }

    getResult() {
        return {
            blablacar: this.blablacar,
        };
    }

    start(poll = false) {
        if (this.done) {
            this.callback(null, this.getResult());

            return false;
        }

        if (this.stopped) {
            return false;
        }

        this.timeStart = moment();

        this.getPrices(poll)
            .then(() => {
                this.done = true;

                if (!this.stopped) {
                    this.callback(null, this.getResult());
                }
            })
            .catch(this.callback);

        return this.needRequestPrices();
    }

    stop() {
        this.stopped = true;
    }
}
