import deepEqual from 'deep-equal';
import {momentTimezone as moment} from '../../../reexports';
import uniqBy from 'lodash/uniqBy';

import {ROBOT} from '../date/formats';

import {TransportType} from '../transportType';

import {getStationFrom, getStationTo} from '../thread/thread';
import {getStationPointById, getCityPointById} from '../point/pointType';
import joinDynamicPriceAnswers from '../search/joinDynamicPriceAnswers';
import joinStaticPriceAnswers from '../search/joinStaticPriceAnswers';
import {getIncrementalTimeoutPromise} from '../timeoutPromise';
import patchSegments from '../segments/patchSegments';
import updateSegments from '../segments/updateSegments';
import metaCreator from '../../actions/search/helpers/metaCreator';
import {
    sortTariffClassKeys,
    getBaseTariffClassKeys,
} from '../segments/getBaseTariffClassKeys';
import noop from '../noop';
import addTariffsToSegments from '../segments/addTariffsToSegments';
import getTariffMatchWeigh from '../segments/getTariffMatchWeigh';
import getSegmentUniqId from '../segments/getSegmentUniqId';

const timeout = getIncrementalTimeoutPromise();

export default class ThreadPricePolling {
    constructor({store, api, callback = noop}) {
        this.api = api;
        this.callback = callback;
        this.context = this.getContext(store);
        this.timeStart = null;
        this.priceAnswers = {
            segments: [],
            tariffs: [],
            querying: {},
        };
        this.segments = [];
        this.stopped = false;
        this.done = false;
    }

    getContext(store) {
        const {
            thread,
            flags,
            nationalVersion,
            language,
            currencies,
            tld,
            isTouch,
            environment,
        } = store.getState();
        const {
            id,
            transportType,
            stations,
            stationFrom,
            departureFrom,
            stationTo,
            tariffsKeys,
            isToCitySearchContext,
        } = thread;

        return {
            currencies,
            flags,
            nationalVersion,
            tld,
            isTouch,
            environment,
            language,
            id,
            transportType,
            stations,
            stationFrom,
            departureFrom,
            stationTo,
            tariffsKeys,
            isToCitySearchContext,
        };
    }

    requestPricesIsPossible() {
        const {
            nationalVersion,
            language,
            transportType,
            stations,
            currencies,
            id,
            tariffsKeys,
            tld,
            isTouch,
            environment,
        } = this.context;

        return Boolean(
            nationalVersion &&
                language &&
                transportType &&
                stations.length &&
                currencies &&
                id &&
                tariffsKeys.length &&
                tld &&
                typeof isTouch === 'boolean' &&
                environment,
        );
    }

    getMetaForSetPrices() {
        const {tld, flags, isTouch, currencies, environment} = this.context;

        return metaCreator(null, {
            tld,
            flags,
            isTouch,
            searchForm: {},
            currencies,
            search: {
                context: {
                    from: {},
                    to: {},
                    time: {
                        now: new Date().valueOf(),
                    },
                },
            },
            environment,
        });
    }

    requestPrices({poll = false, req}) {
        const queries = [];

        if (!this.requestPricesIsPossible()) {
            return Promise.resolve(queries);
        }

        const {
            flags,
            environment: {type: environmentType},
            nationalVersion,
            language,
            transportType,
            stations,
            isToCitySearchContext,
        } = this.context;
        const stationFromData = getStationFrom({stations});
        const stationToData = getStationTo({
            stations,
            stationFrom: stationFromData,
        });
        const pointFrom = getStationPointById(stationFromData.id);
        const pointTo = getStationPointById(stationToData.id);
        const now = moment().valueOf();
        const departureFromMoment = moment.parseZone(
            stationFromData.departureLocalDt,
        );

        if (this.needRequestTrainTariffs()) {
            queries.push(
                this.api.exec(
                    'trainTariffs2',
                    {
                        transportType,
                        pointFrom,
                        pointTo,
                        now,
                        language,
                        nationalVersion,
                        flags,
                        poll,
                        environmentType,
                        startTime: stationFromData.departureLocalDt,
                        endTime: departureFromMoment.add(1, 'minutes').format(),
                    },
                    req,
                ),
            );
        }

        if (this.needRequestStaticBusTariffs()) {
            const startDate = departureFromMoment.format(ROBOT);
            const endDate = departureFromMoment.add(1, 'days').format(ROBOT);

            queries.push(
                this.api.exec(
                    'staticTariffs2',
                    {
                        language,
                        nationalVersion,
                        pointFrom,
                        pointTo,
                        startDate,
                        endDate,
                        transportType: TransportType.bus,
                    },
                    req,
                ),
            );

            queries.push(
                this.api.execBusTariffs2(
                    {
                        flags,
                        isCitySearch: isToCitySearchContext,
                        context: {
                            language,
                            from: {
                                key: getCityPointById(
                                    stationFromData.settlement.id,
                                ),
                                slug: stationFromData.slug,
                            },
                            to: {
                                key: isToCitySearchContext
                                    ? getCityPointById(
                                          stationToData.settlement.id,
                                      )
                                    : pointTo,
                                slug: stationToData.slug,
                            },
                            when: {
                                date: startDate,
                                nextDate: endDate,
                            },
                        },
                        nationalVersion,
                    },
                    req,
                ),
            );
        }

        return Promise.all(queries);
    }

    /**
     * Проверяет эквивалентность контекста, с которым был создан поллинг, и переданного
     * @param {Object} store
     * @return {boolean}
     */
    isEqualContext(store) {
        return deepEqual(this.context, this.getContext(store));
    }

    /**
     * Проверяет актуальность полинга по времени
     * @return {boolean}
     */
    isActual() {
        return !this.timeStart || this.timeStart > moment().add(-5, 'minutes');
    }

    needRequestTrainTariffs() {
        const {transportType} = this.context;
        const {train: trainQuerying} = this.priceAnswers.querying;

        return (
            (transportType === TransportType.train ||
                transportType === TransportType.suburban) &&
            (typeof trainQuerying === 'undefined' || trainQuerying === true)
        );
    }

    needRequestStaticBusTariffs() {
        const {transportType} = this.context;

        return (
            transportType === TransportType.bus &&
            typeof this.priceAnswers.querying.static === 'undefined'
        );
    }

    needRequestPrices() {
        if (this.stopped || this.done) {
            return false;
        }

        return (
            this.requestPricesIsPossible() &&
            (this.needRequestTrainTariffs() ||
                this.needRequestStaticBusTariffs())
        );
    }

    _setPriceAnswers(answers) {
        const {currencies} = this.context;
        const {segments, querying} = joinDynamicPriceAnswers(
            answers,
            currencies,
        );
        const tariffs = joinStaticPriceAnswers(answers);
        const meta = this.getMetaForSetPrices();

        const uniqSegments = uniqBy(
            [...this.priceAnswers.segments, ...segments].map(segment => {
                if (segment.id) {
                    return segment;
                }

                return {
                    ...segment,
                    id: getSegmentUniqId(segment),
                };
            }),
            'id',
        );

        this.priceAnswers = {
            ...this.priceAnswers,
            segments: patchSegments({
                segments: uniqSegments,
                meta,
                isDynamic: true,
            }),
            querying: {
                ...this.priceAnswers.querying,
                ...querying,
                static: false,
            },
            tariffs: [...this.priceAnswers.tariffs, ...tariffs],
        };
    }

    getPrices(poll = false, numberRetry = 0) {
        if (!this.needRequestPrices()) {
            return Promise.resolve(this.priceAnswers);
        }

        numberRetry++;

        return this.requestPrices({poll}).then(answers => {
            this._setPriceAnswers(answers);

            if (this.needRequestPrices()) {
                return timeout(numberRetry).then(() =>
                    this.getPrices(true, numberRetry),
                );
            }

            return this.priceAnswers;
        });
    }

    setPrices() {
        const meta = this.getMetaForSetPrices();
        const {segments, tariffs} = this.priceAnswers;
        const anySegmentCanSupplySegments = segments.some(
            segment => segment.canSupplySegments === false,
        );

        if (tariffs.length || anySegmentCanSupplySegments) {
            this.segments = [this.getFakeSegment()];
        }

        // матчим тарифы
        this.segments = addTariffsToSegments(this.segments, tariffs, meta);

        this.segments = updateSegments(this.segments, segments, meta);

        this.segments = this.segments.map(segment => {
            segment.tariffClassKeys = sortTariffClassKeys({
                tariffClassKeys: getBaseTariffClassKeys(segment),
                segment,
            });

            return segment;
        });
    }

    getMatchWeight(segment) {
        const {tariffsKeys} = this.context;
        const {keys, key} = segment;

        return getTariffMatchWeigh({tariffsKeys}, {keys, key});
    }

    getSegment() {
        if (this.segments.length === 0) {
            return null;
        }

        const match = this.segments
            .filter(
                segment =>
                    segment.tariffs &&
                    segment.tariffs.classes &&
                    Object.keys(segment.tariffs.classes).length,
            )
            .map(segment => {
                const matchWeight = this.getMatchWeight(segment);

                return {
                    matchWeight,
                    segment,
                };
            })
            .reduce(
                (bestMatch, element) => {
                    if (element.matchWeight > bestMatch.matchWeight) {
                        return element;
                    }

                    return bestMatch;
                },
                {matchWeight: 0},
            );

        if (match.matchWeight) {
            return match.segment;
        }

        return null;
    }

    /**
     * Возвращает фэйковый сегмент для случая, когда есть тарифы, но нет сегментов
     * @return {Object}
     */
    getFakeSegment() {
        const {tariffsKeys, transportType} = this.context;

        return {
            source: 'fake',
            tariffsKeys,
            transport: {code: transportType},
            stationFrom: {},
            stationTo: {},
        };
    }

    getResult() {
        return {
            segment: this.getSegment(),
        };
    }

    /**
     * Стартует полинг цен
     * @param {boolean} poll - признак старта опроса цен в режиме полинга
     * @return {boolean} - признак, что будет произведен запрос к апи за ценами
     */
    start(poll = false) {
        if (this.done) {
            this.callback(null, this.getResult());

            return false;
        }

        if (this.stopped) {
            return false;
        }

        this.timeStart = moment();

        this.getPrices(poll)
            .then(() => {
                this.setPrices();

                this.done = true;

                if (!this.stopped) {
                    this.callback(null, this.getResult());
                }
            })
            .catch(this.callback);

        return this.needRequestPrices();
    }

    stop() {
        this.stopped = true;
    }
}
