import groupBy from 'lodash/groupBy';
import {fixQuotes} from './fix-quotes';

import {
    STATIONS_TYPES_ORDER,
    PLANE_TYPE,
    TRANSPORT_TYPES_WITH_DISPLAYED_STATIONS,
} from './transportType';

export const stationTypeToMetrikaTarget = {
    train: 'morda_railway_stations_link_click',
    plane: 'morda_airports_link_click',
    bus: 'morda_bus_stations_link_click',
    water: 'morda_water_stations_link_click',
    aeroexpress: 'morda_aeroexpress_link_click',
};

// Типы станций, которые допустимо использовать в качестве заголовка станции,
// если название станции совпадает с названием города
const stationTypeCodes = new Set([
    'train_station',
    'airport',
    'bus_station',
    'port',
    'river_port',
    'marine_station',
]);

/**
 * Возвращает массив станций с установленным полем originalTitle
 * @param {Object} station - массив станций
 * @return {Object[]}
 */
export function setOriginalTitle(station) {
    station.originalTitle = station.title;

    return station;
}

/**
 * Возвращает заголовок для одиночной станции в группе
 * @param {Object} station - станция
 * @param {string} cityTitleLowerCase - название города в нижнем регистре
 * @return {string}
 */
export function getSingleStationTitle(station, cityTitleLowerCase) {
    const stationTitleLowerCase = station.title.toLowerCase();
    const stationTypeTitleToLowerCase =
        station.station_type.title.toLowerCase();

    if (
        stationTypeCodes.has(station.station_type.code) &&
        (stationTitleLowerCase === cityTitleLowerCase ||
            stationTitleLowerCase ===
                `${stationTypeTitleToLowerCase} ${cityTitleLowerCase}`)
    ) {
        return station.station_type.title;
    }

    return station.title;
}

/**
 * Устанавливает значения метрик в зависомости от типа станции
 * @param {Object} station
 * @return {Object}
 */
export function setMetriks(station) {
    station.metrikaTarget = stationTypeToMetrikaTarget[station.t_type];

    if (station.t_type === PLANE_TYPE) {
        station.aeroexpressMetrikaTarget =
            stationTypeToMetrikaTarget.aeroexpress;
    }

    return station;
}

/**
 * Исправляет название станций метро и исправляет кавычки
 * @param {Object} station
 * @return {Object}
 */
export function setFinalTitle(station) {
    if (station.is_metro) {
        station.title = station.title.replace(/^\S\.\s/, '');
    }

    station.title = fixQuotes(station.title);

    return station;
}

/**
 * Возвращает массив групп станций, сгруппированных по типу станции
 * и в приоретизированном порядке (STATIONS_TYPES_ORDER)
 * @param {*} data
 * @param {*} city
 * @return {Object[]}
 */
export function prepareStationsForIndexPage(data, city) {
    const stations = data.map(setOriginalTitle);
    const cityTitleLowerCase = city.title.toLowerCase();
    const byType = groupBy(stations, 't_type');

    Object.keys(byType).forEach(transportType => {
        const group = byType[transportType];

        if (group.length === 1) {
            const station = group[0];

            station.title = getSingleStationTitle(station, cityTitleLowerCase);
        }
    });

    const byTitle = groupBy(stations, station => station.title.toLowerCase());

    Object.keys(byTitle).forEach(title => {
        const group = byTitle[title].filter(station =>
            TRANSPORT_TYPES_WITH_DISPLAYED_STATIONS.includes(station.t_type),
        );

        if (title === cityTitleLowerCase || group.length > 1) {
            group.forEach(station => {
                if (
                    station.station_type.code !== 'station_without_type' &&
                    station.title !== station.station_type.title
                ) {
                    station.title = `${station.station_type.title} ${station.title}`;
                }
            });
        }
    });

    stations.map(setMetriks).map(setFinalTitle);

    return STATIONS_TYPES_ORDER.filter(
        transportType => byType[transportType],
    ).map(transportType => ({
        t_type: transportType,
        stations: byType[transportType],
    }));
}
