import {momentTimezone as moment} from '../../../reexports';

import {SPECIAL_VALUES} from '../date/values';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';

import {getNextDayFormatted} from '../date/utils';

export function getContextFromForm(searchForm) {
    const {
        transportType,
        from,
        originalFrom,
        to,
        originalTo,
        when,
        searchNext,
        language,
        time,
        userInput,
        plan,
        sameSuburbanZone,
        distance,
    } = searchForm;

    const {timezone, now} = time;
    const searchForPastDate =
        !searchNext && when.date
            ? moment
                  .tz(when.date, timezone)
                  .isBefore(moment.tz(now, timezone), 'day')
            : false;

    const context = {
        userInput,
        transportType,
        from,
        originalFrom,
        to,
        originalTo,
        searchNext,
        when,
        time,
        language,
        searchForPastDate,
        sameSuburbanZone,
        distance,
    };

    if (plan) {
        context.plan = plan;
    }

    if (when.date) {
        context.when = {
            ...when,
            nextDate: getNextDayFormatted(when.date, time.timezone),
        };
    }

    return context;
}

export function getSearchContext(searchForm, search) {
    return {
        ...getContextFromForm(searchForm),
        transportTypes: search.context.transportTypes,
        latestDatetime: search.context.latestDatetime,
        canonical: search.canonical || null,
    };
}

/**
 * Возвращает измененный контекст
 * Данные о исходном контексте хранятся для построения дисклеймеров сужения/расширения
 * @param { Object } formContext - контекст полученный из поисковой формы
 * @param { Object } searchContext - контекст полученный из ответа ручки бекенда
 * @return { Object }
 */
export function getExtendedSearchContext(formContext, searchContext) {
    const {search, original} = searchContext;

    return {
        ...formContext,
        isChanged: true,
        from: {
            ...formContext.from,
            ...search.pointFrom,
        },
        to: {
            ...formContext.to,
            ...search.pointTo,
        },
        original,
    };
}

/**
 * Функция возвращает признак того, нужно ли использовать пригородный поисковый контекст
 * @param { Object } from - контекст пункта отправления
 * @param { Object } to - контекст пункта прибытия
 * @param { boolean } searchNext - признак поиска ближайших рейсов
 * @return { Object }
 */
export function useSuburbanSearchContext({from, to, searchNext}) {
    return (
        searchNext &&
        !from.title &&
        !from.key &&
        !from.slug &&
        !to.title &&
        !to.key &&
        !to.slug
    );
}

/**
 * Валидирует поле when поискового контекста
 * @param {Object} when
 * @return {*}
 */
export function validateWhen(when) {
    if (
        when.date ||
        when.special ||
        when.weekday !== undefined ||
        when.text === ''
    ) {
        return [];
    }

    return [
        {
            fields: ['when'],
            type: 'incorrect',
        },
    ];
}

/**
 * Возвращает параметр when для контекста поиска на все дни
 * @param {string} language
 * @return {{text: *, special: string}}
 */
export function getWhenForAllDays(language) {
    return {
        text: SPECIAL_VALUES[language][DateSpecialValue.allDays],
        special: DateSpecialValue.allDays,
    };
}

/**
 * Возвращает параметр when для контектса поиска на сегодня
 * @param {string} language
 * @return {{text: *, special: string}}
 */
export function getWhenForToday(language) {
    return {
        text: SPECIAL_VALUES[language][DateSpecialValue.today],
        special: DateSpecialValue.today,
    };
}
