import CurrencyCode from '../../interfaces/CurrencyCode';

import convertPrice from '../currencies/convertPrice';
import isPriceGreater from '../segments/isPriceGreater';

const PRICE_LIMIT = {
    currency: CurrencyCode.rub,
    value: 10,
};

/**
 * Отфильтровывает некорректные цены в сегментах и сегменты не содержащие корректных цен
 * @param {Object[]} segments - сегменты с ценами
 * @param {Object} currencies - данные валют
 * @return {Object[]}
 */
export default function filterSegmentsWithIncorrectPrice(segments, currencies) {
    const nationalCurrency = {
        ...currencies,
        preferredCurrency: CurrencyCode.rub,
    };

    return segments.reduce((filteredSegments, segment) => {
        const {tariffs} = segment;

        if (tariffs) {
            const {classes} = tariffs || {};
            const filteredClasses = Object.keys(classes).reduce(
                (result, key) => {
                    const {price} = classes[key];

                    if (
                        price &&
                        isPriceGreater(
                            convertPrice(price, nationalCurrency),
                            PRICE_LIMIT,
                        )
                    ) {
                        result[key] = classes[key];
                    }

                    return result;
                },
                {},
            );

            if (Object.keys(filteredClasses).length) {
                filteredSegments.push({
                    ...segment,
                    tariffs: {
                        ...segment.tariffs,
                        classes: filteredClasses,
                    },
                });
            }
        }

        return filteredSegments;
    }, []);
}
