import {YBUS} from './tariffSources';
import cloneDeep from 'lodash/cloneDeep';

/**
 * Метод собирает вариант предложения автобусного партнера (соединяет данные о тарифе и данные о партнере)
 * @param tariff - тариф на основе которого собирается вариант
 * @returns {Object}
 */

export function buildVariant(tariff) {
    if (!tariff) {
        return null;
    }

    const {classes, supplier} = tariff;
    const {bus} = classes;
    const variant = {...bus};

    if (supplier) {
        variant.supplier = supplier;
    }

    return variant;
}

/**
 * Метод получает ключ в объекте вариантов по тарифу
 * @param {Object} tariffs
 * @return {string}
 */
export function getVariantKey(tariffs) {
    if (!tariffs) {
        return null;
    }

    const {supplier} = tariffs;
    const {code} = supplier || {};

    // Если нет кода, значит это Я.Автобусы
    return code || YBUS;
}

/**
 * Создает новый объект для вариантов на основе тарифа
 * @param {Object} tariffs
 * @return {Object}
 */
export function getNewVariantsObject(tariffs) {
    if (!tariffs) {
        return {};
    }

    return {[getVariantKey(tariffs)]: buildVariant(tariffs)};
}

function shouldAddVariants({segment, variantSourceSegment}) {
    // Нет смысла добавлять варианты, если у одного из сегментов нет тарифов или если сегменты одинаковые или у сегмента нет ссылки на покупку
    return Boolean(
        segment.tariffs &&
            variantSourceSegment.tariffs &&
            segment !== variantSourceSegment &&
            segment.tariffs.classes.bus &&
            variantSourceSegment.tariffs.classes.bus &&
            variantSourceSegment.tariffs.classes.bus.orderUrl,
    );
}

/**
 * Метод добавляет в тарифы сегмента автобусов варианты на основе тарифов двух сегментов
 * @param {Object} segment - сегмент в который дописываются варианты
 * @param {Object} variantSourceSegment - добавляемый сегмент
 * @param {boolean} overwrite - определяет, можно ли перезаписывать вариант с тем же ключом
 * @return {Object}
 */
export function addVariantsToBusSegment(
    segment,
    variantSourceSegment,
    overwrite = true,
) {
    if (!shouldAddVariants({segment, variantSourceSegment})) {
        return segment;
    }

    const result = cloneDeep(segment);
    const additionalSegment = cloneDeep(variantSourceSegment);
    const {tariffs} = result;
    const {tariffs: additionalTariffs} = additionalSegment;

    let variants =
        tariffs.classes.bus.variants || getNewVariantsObject(tariffs);

    // Если в добавляемом сегменте уже есть варианты, то просто вливаем их в основной сегмент
    if (additionalSegment.hasVariants) {
        variants = Object.assign(
            additionalTariffs.classes.bus.variants,
            variants,
        );
    } else {
        const variantKey = getVariantKey(additionalTariffs);

        // Перезаписываем значение по ключу только если разрешено или оно ещё не задано
        if (overwrite || !variants[variantKey]) {
            variants[variantKey] = buildVariant(additionalTariffs);
        }
    }

    // Если вариант оказался только один, то это равносильно отсутствию вариантов
    if (Object.keys(variants).length > 1) {
        result.tariffs.classes.bus.variants = variants;
        result.hasVariants = true;
    }

    return result;
}
