import {TransportType, FilterTransportType} from '../transportType';
import ISegment from '../../interfaces/segment/ISegment';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import SearchSegment from '../../interfaces/state/search/SearchSegment';
import IState from '../../interfaces/state/IState';

import isTransfer from './isTransfer';

type Segment = ISegment | ITransfer | ITransferSegment;

interface IAccInterface {
    transportTypes: Set<TransportType>;
    hasElectronicTicket: boolean;
    hasDynamicPricing: boolean;
}

interface IAccumulatorFunction {
    (acc: IAccInterface, segment: Segment): IAccInterface;
}

const analyseTransportTypes: IAccumulatorFunction = (acc, {transport}) => {
    acc.transportTypes.add(transport.code);

    return acc;
};

const analyseElectronicTicket: IAccumulatorFunction = (acc, {tariffs}) => {
    acc.hasElectronicTicket =
        acc.hasElectronicTicket || Boolean(tariffs && tariffs.electronicTicket);

    return acc;
};

const analyseDynamicPricing: IAccumulatorFunction = (acc, segment) => {
    if (isTransfer(segment)) {
        return acc;
    }

    acc.hasDynamicPricing =
        acc.hasDynamicPricing || Boolean(segment.hasDynamicPricing);

    return acc;
};

interface IAnalyzeSegmentsResult {
    segments: SearchSegment[];
    transportTypes: TransportType[];
    isSuburbanSearchResult: boolean;
    isBusSearchResult: boolean;
    stats: {
        hasElectronicTicket: boolean;
        hasDynamicPricing: boolean;
    };
}

// Функция анализирует выдачу сегментов после диспатча данных но перед обработкой в reducer
export default function analyseSegments(
    segments: SearchSegment[],
    state: IState,
): IAnalyzeSegmentsResult {
    const {context} = state.search;
    const {transportType} = context;

    const analysers = [
        analyseTransportTypes,
        analyseElectronicTicket,
        analyseDynamicPricing,
    ];

    // Одним проходом проверяем все сегменты и собираем данные
    const aggregatedInfo = segments.reduce(
        (acc, segment) => {
            analysers.forEach(analyser => {
                acc = analyser(acc, segment);
            });

            if (isTransfer(segment)) {
                segment.segments.forEach(transferSegment => {
                    analysers.forEach(analyser => {
                        acc = analyser(acc, transferSegment);
                    });
                });
            }

            return acc;
        },
        {
            transportTypes: new Set() as Set<TransportType>,
            hasElectronicTicket: false,
            hasDynamicPricing: false,
        },
    );

    const transportTypes = [...aggregatedInfo.transportTypes]; // Превращаем Set в массив
    const {hasDynamicPricing, hasElectronicTicket} = aggregatedInfo;

    const isSuburbanSearchResult =
        transportType === FilterTransportType.suburban ||
        (transportType === FilterTransportType.all &&
            transportTypes.length > 0 &&
            transportTypes.every(
                transport => transport === TransportType.suburban,
            ));

    const isBusSearchResult =
        transportType === FilterTransportType.bus ||
        (transportType === FilterTransportType.all &&
            transportTypes.length > 0 &&
            transportTypes.every(transport => transport === TransportType.bus));

    return {
        segments,
        transportTypes,
        isSuburbanSearchResult,
        isBusSearchResult,
        stats: {
            hasElectronicTicket,
            hasDynamicPricing,
        },
    };
}
