import isEmpty from 'lodash/isEmpty';
import assign from 'lodash/assign';

import {RATION_TO_OFF_BADGES} from './constants';

import findCheapestValue from './cheapest/findCheapestValue';
import checkIsManyCheapest from './cheapest/checkIsManyCheapest';

import findFastestSegmentValue from './fastest/findFastestSegmentValue';
import checkIsManyFastest from './fastest/checkIsManyFastest';
import ISegment from 'common/interfaces/segment/ISegment';

export interface IBadgesData {
    /** Значение минимальной цены среди всех сегментов */
    cheapestValue?: number;
    /** true, если критическая доля сегментов определяется, как "самые дешевые" */
    isManyCheapest?: boolean;
    /** значение времени в пути самого быстрого сегмента */
    fastestSegmentValue?: number;
    /** true, если критическая доля сегментов определяется, как "самые быстрые" */
    isManyFastest?: boolean;
}

/**
 * Формирует данные, необходимые для получения бейджиков
 * @param {Object} segments — все сегменты текущей выдачи
 * @return {Object}
 */
export default function getBadgesData(
    segments: ISegment[],
): IBadgesData | null {
    const cheapestValue = findCheapestValue(segments);
    const fastestSegmentValue = findFastestSegmentValue(segments);

    const badgesData: IBadgesData = {};

    if (isFinite(cheapestValue)) {
        /* Проверяем, много ли сегментов являются "дешевыми" или "самыми дешевыми" */
        const isManyCheapest = checkIsManyCheapest(
            segments,
            cheapestValue,
            RATION_TO_OFF_BADGES,
        );

        assign(badgesData, {
            isManyCheapest,
            cheapestValue,
        });
    }

    if (fastestSegmentValue) {
        const isManyFastest = checkIsManyFastest(
            segments,
            fastestSegmentValue,
            RATION_TO_OFF_BADGES,
        );

        assign(badgesData, {
            isManyFastest,
            fastestSegmentValue,
        });
    }

    return isEmpty(badgesData) ? null : badgesData;
}
