import moment, {Moment} from '../../../reexports/moment-timezone';

import ISegment from '../../interfaces/segment/ISegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ISubSegment from '../../interfaces/segment/ISubSegment';

type Segment = ISegment | ITransfer | ITransferSegment | ISubSegment;

type TimeField = 'departure' | 'arrival';

type CacheTimezone = Partial<Record<string, Moment>>;

type CacheElement = Partial<Record<TimeField, CacheTimezone>>;

const cache = new WeakMap<Segment, CacheElement>();

export function getLocalTime(
    segment: Segment,
    field: TimeField,
    timezone: string,
): Moment {
    const inCache = cache.get(segment);

    if (!inCache) {
        const momentObj = moment.tz(segment[field], timezone);
        const cacheElement: CacheElement = {
            [field]: {
                [timezone]: momentObj,
            },
        };

        cache.set(segment, cacheElement);

        return momentObj;
    }

    const inCacheField = inCache[field];

    if (!inCacheField) {
        const momentObj = moment.tz(segment[field], timezone);

        inCache[field] = {
            [timezone]: momentObj,
        };

        return momentObj;
    }

    const inCacheTimezone = inCacheField[timezone];

    if (!inCacheTimezone) {
        const momentObj = moment.tz(segment[field], timezone);

        inCacheField[timezone] = momentObj;

        return momentObj;
    }

    return inCacheTimezone;
}

interface IGetSegmentTimesResult {
    departureMoment: Moment;
    arrivalMoment: Moment;
    departureRailwayMoment: Moment | null;
    arrivalRailwayMoment: Moment | null;
}

// Вернет true для пар Москва - Мытищи, Казань - Рязань, Астана - Саратов, Ташкент - Москва
// Вернет false для пар Москва - Екатеринбург, Омск - Тюмень, Актобе - Москва, Ташкент - Екатеринбург
function bothStationsHaveRailwayTimezone({
    departureMoment,
    arrivalMoment,
    departureRailwayMoment,
    arrivalRailwayMoment,
}: IGetSegmentTimesResult): boolean {
    if (!departureRailwayMoment || !arrivalRailwayMoment) {
        return false;
    }

    return (
        departureMoment.hours() === departureRailwayMoment.hours() &&
        departureMoment.minutes() === departureRailwayMoment.minutes() &&
        arrivalMoment.hours() === arrivalRailwayMoment.hours() &&
        arrivalMoment.minutes() === arrivalRailwayMoment.minutes()
    );
}

export function getSegmentTimes(segment: Segment): IGetSegmentTimesResult {
    const departureRailwayTimezone = segment.stationFrom.railwayTimezone;
    const arrivalRailwayTimezone = segment.stationTo.railwayTimezone;
    const result = {
        departureMoment: getLocalDepartureTime(segment),
        arrivalMoment: getLocalArrivalTime(segment),
        departureRailwayMoment: departureRailwayTimezone
            ? getLocalTime(segment, 'departure', departureRailwayTimezone)
            : null,
        arrivalRailwayMoment: arrivalRailwayTimezone
            ? getLocalTime(segment, 'arrival', arrivalRailwayTimezone)
            : null,
    };

    if (bothStationsHaveRailwayTimezone(result)) {
        // Тут какие-то адские костыли для отображения временных зон на десктопе
        result.departureRailwayMoment = null;
        result.arrivalRailwayMoment = null;
    }

    return result;
}

export function getDepartureTime(segment: Segment, timezone: string): Moment {
    return getLocalTime(segment, 'departure', timezone);
}

export function getArrivalTime(segment: Segment, timezone: string): Moment {
    return getLocalTime(segment, 'arrival', timezone);
}

export function getLocalDepartureTime(segment: Segment): Moment {
    return getDepartureTime(segment, segment.timezoneFrom);
}

export function getLocalArrivalTime(segment: Segment): Moment {
    return getArrivalTime(segment, segment.timezoneTo);
}
