import intersection from 'lodash/intersection';

import isPriceGreater from './isPriceGreater';
import getLowestPrice from './getLowestPrice';
import {makeCacheable} from '../cache';
import priceRangeManager from '../filters/priceRange';

/**
 * Метод получения минимальной цены среди результатов поиска,
 * с указанием типа транспорта
 * @param {Object[]} segments - массив сегментов
 * @param {Object} [options={}] - опции для получения минимальной цены
 * @param {string[]} [options.tariffClassKeys=[]] - настройка для поиска по определенным тарифам
 * @param {Array} [options.priceRangeFilterValue=[]] - значение фильтра priceRange, для того, чтобы минимальная цена
 * соотвествовала интервалам фильтра
 * @return {Object}
 */
const gatherMinPriceData = (
    segments,
    {tariffClassKeys = [], priceRangeFilterValue = []} = {},
) =>
    segments
        .filter(({isGone}) => !isGone)
        .reduce((result, segment) => {
            let requiredTariffClassKeys = tariffClassKeys;

            if (priceRangeFilterValue.length) {
                // получаем тарифы в сегмента, подходящие по фильтру "Цены"
                const suitableByFilterTariffsKeys =
                    priceRangeManager.getSuitableSegmentTariffClasses(
                        priceRangeFilterValue,
                        segment,
                    );

                const suitableTariffClassKeys = tariffClassKeys.length
                    ? intersection(tariffClassKeys, suitableByFilterTariffsKeys)
                    : suitableByFilterTariffsKeys;

                if (!suitableTariffClassKeys.length) {
                    return result;
                }

                requiredTariffClassKeys = suitableTariffClassKeys;
            }

            const price = getLowestPrice(segment, requiredTariffClassKeys);

            if (!price) {
                return result;
            }

            if (!result || isPriceGreater(result.price, price)) {
                return {transportType: segment.transport.code, price};
            }

            return result;
        }, null);

export default makeCacheable(gatherMinPriceData);
