import deepEqual from 'deep-equal';
import intersection from 'lodash/intersection';

import {YBUS} from './tariffSources';
import {SORTING_TYPES} from '../sort/utils';
import {
    BUS_TYPE,
    PLANE_TYPE,
    TRAIN_TYPE,
    SUBURBAN_TYPE,
} from '../transportType';
import {
    PLANE_TARIFF_CLASSES,
    TRAIN_TARIFF_CLASSES,
    TARIFF_CLASSES_BY_TYPE,
    TRAIN_TARIFF_CLASSES_DAY,
} from './tariffClasses';
import isOvernight from './isOvernight';

const MIN_NIGHT_TRAIN_DURATION = 60 * 60 * 3;

function pickPresent(keys, obj) {
    return keys.filter(key => obj[key] !== undefined);
}

function pickFirst(keys, obj) {
    const first = keys.find(key => obj[key] !== undefined);

    return first ? [first] : [];
}

/**
 * Метод вернёт классы тарифов сегмента
 * @param {Object} segment - данные сегмента
 * @return {string[]}
 */
export function getBaseTariffClassKeys(segment) {
    const {
        tariffs,
        transport,
        hasTrainTariffs,
        queryingPrices = false,
    } = segment;
    const classes = tariffs && tariffs.classes;

    if (!classes) {
        return [];
    }

    // Если у электрички есть поездатые цены, то будем отображать ее тарифы как поездатые
    const transportType = hasTrainTariffs ? TRAIN_TYPE : transport.code;

    switch (transportType) {
        case BUS_TYPE:
        case SUBURBAN_TYPE:
            return pickPresent(TARIFF_CLASSES_BY_TYPE[transport.code], classes);
        case PLANE_TYPE:
            return pickFirst(PLANE_TARIFF_CLASSES, classes);
        case TRAIN_TYPE: {
            const result = pickPresent(TRAIN_TARIFF_CLASSES, classes).filter(
                classKey => classes[classKey].seats !== 0,
            );

            if (
                result.length === 0 &&
                transportType !== transport.code &&
                queryingPrices === false
            ) {
                // Если поездатые тарифы так и не приехали, но есть электричечные, то вернем их
                const resultAsRealTransportType = pickPresent(
                    TARIFF_CLASSES_BY_TYPE[transport.code],
                    classes,
                );

                if (resultAsRealTransportType.length) {
                    return resultAsRealTransportType;
                }
            }

            return result;
        }

        default:
            return Object.keys(classes);
    }
}

/**
 * Метод вернёт классы тарифов автобусного сегмента с учетом наличия мест
 * @param {Object} segment - данные сегмента
 * @return {string[]}
 */
export function getBusTariffClassKeys(segment) {
    const baseKeys = getBaseTariffClassKeys(segment);

    return segment.source === YBUS
        ? baseKeys.filter(
              classKey => segment.tariffs.classes[classKey].seats !== 0,
          )
        : baseKeys;
}

/**
 * Возвращает список тарифов для отображения подходящих под фильтр trainTariffClass
 * @param {Array} tariffClassKeys
 * @param {Object} segment
 * @param {Object} filters
 * @return {Array}
 */
export function filterTariffClassKeysByTrainTariffClass(
    tariffClassKeys,
    segment,
    filters,
) {
    const {transport} = segment;

    if (tariffClassKeys.length === 0 || transport.code !== TRAIN_TYPE) {
        return tariffClassKeys;
    }

    const filteredTrainTariffClasses =
        filters && filters.trainTariffClass && filters.trainTariffClass.value;

    if (filteredTrainTariffClasses && filteredTrainTariffClasses.length) {
        tariffClassKeys = intersection(
            tariffClassKeys,
            filteredTrainTariffClasses,
        );
    }

    return tariffClassKeys;
}

/**
 * Возвращает список тарифов для отображения.
 * Используется в отображении сегментов на поиске: если отфильтрованы плацкартные места,
 * то их стоимость не показывается в сниппете
 * @param {Object} obj
 * @param {Array} obj.tariffClassKeys - начальный список тарифов сегмента
 * @param {Object} obj.segment
 * @param {Object} [obj.filtering]
 * @return {Array<string>}
 */
export function filterTariffClassKeys({tariffClassKeys, segment, filtering}) {
    if (!filtering || !filtering.filters) {
        return tariffClassKeys;
    }

    return filterTariffClassKeysByTrainTariffClass(
        tariffClassKeys,
        segment,
        filtering.filters,
    );
}

/**
 * Сортирует классы тарифов сегмента
 * @param {Object} tariffClassKeys
 * @param {Object} segment
 * @param {Object} [sort]
 * @return {Object}
 */
export function sortTariffClassKeys({tariffClassKeys, segment, sort}) {
    const {tariffs, transport} = segment;

    if (tariffClassKeys.length === 0 || transport.code !== 'train') {
        return tariffClassKeys;
    }

    if (sort && sort.by === SORTING_TYPES.PRICE) {
        const classes = tariffs && tariffs.classes;
        const multiplier = sort.reverse ? -1 : 1;

        return tariffClassKeys.sort((a, b) =>
            classes[a].price && classes[b].price
                ? (classes[a].price.value - classes[b].price.value) * multiplier
                : 0,
        );
    }

    const trainTariffClasses = intersection(
        TRAIN_TARIFF_CLASSES,
        tariffClassKeys,
    );
    const trainTariffClassesDay = intersection(
        TRAIN_TARIFF_CLASSES_DAY,
        tariffClassKeys,
    );

    if (
        deepEqual(trainTariffClasses, trainTariffClassesDay) ||
        (isOvernight(segment) && segment.duration > MIN_NIGHT_TRAIN_DURATION)
    ) {
        return trainTariffClasses;
    }

    return trainTariffClassesDay;
}
