import {SORTING_TYPES} from '../sort/utils';

import {TransportType} from '../transportType';
import ISegment from '../../interfaces/segment/ISegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ISegmentTariffPriceWithClass from '../../interfaces/segment/ISegmentTariffPriceWithClass';

import {
    getBaseTariffClassKeys,
    getBusTariffClassKeys,
    sortTariffClassKeys,
} from './getBaseTariffClassKeys';

/**
 * Возвращает наименьшую цену среди тарифов сегмента с учетом наличия мест
 * @param {Object} segment - данные сегмента
 * @param {Array} [tariffClassKeys] - список тарифов сегмента, по которым ищутся минимальные цены
 * @return {Object}
 */
export default function getLowestPrice(
    segment: ISegment | ITransfer | ITransferSegment,
    tariffClassKeys: string[] = [],
): ISegmentTariffPriceWithClass | null {
    const sort = {by: SORTING_TYPES.PRICE};

    const allTariffClassKeys =
        segment.transport.code === TransportType.bus
            ? getBusTariffClassKeys(segment)
            : getBaseTariffClassKeys(segment);

    if (!segment.tariffs || !allTariffClassKeys.length) {
        return null;
    }

    const existedTariffClassKeys = tariffClassKeys.length
        ? tariffClassKeys.filter(tariffClassKey =>
              allTariffClassKeys.includes(tariffClassKey),
          )
        : allTariffClassKeys;

    if (!existedTariffClassKeys.length) {
        return null;
    }

    const cheapestTariffKey = sortTariffClassKeys({
        tariffClassKeys: existedTariffClassKeys,
        segment,
        sort,
    })[0];
    const cheapestClass = segment.tariffs.classes[cheapestTariffKey];

    return {
        ...(cheapestClass.nationalPrice || cheapestClass.price),
        class: cheapestTariffKey,
    };
}
