import size from 'lodash/size';

import {YBUS} from './tariffSources';

import HttpMethod from '../../interfaces/HttpMethod';
import {TransportType} from '../transportType';
import ISegment from '../../interfaces/segment/ISegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ISegmentTariffClass from '../../interfaces/segment/ISegmentTariffClass';
import IStateSeoQueryParams from '../../interfaces/state/IStateSeoQueryParams';

import ybusUrl from '../url/ybusUrl';
import updateQuery from '../url/updateQuery';
import getValueFromEnum from '../enum/getValueFromEnum';
import applyUtm from '../url/applyUtm';

interface IGetPurchaseLinkObject {
    segment: ISegment | ITransfer | ITransferSegment;
    tariffClass: ISegmentTariffClass;

    seoQueryParams?: IStateSeoQueryParams;
    clientId?: string;
    directLink?: boolean;
    medium?: string;
}

interface IPurchaseLinkObject {
    method: HttpMethod;
    POSTParams: Record<string, string>;

    href?: string;
}

export default function getPurchaseLinkObject({
    segment,
    tariffClass,
    seoQueryParams,
    clientId,
    directLink = true,
    medium = '',
}: IGetPurchaseLinkObject): IPurchaseLinkObject {
    const {partnerOrderRequest, deepUrl, orderUrl} = tariffClass;
    let href: string | null;
    let method = HttpMethod.get;
    let POSTParams = {};

    if ('source' in segment && segment.source === YBUS) {
        href = ybusUrl(tariffClass, medium);
    } else if (
        segment.transport &&
        segment.transport.code === TransportType.bus &&
        directLink &&
        partnerOrderRequest &&
        partnerOrderRequest.url
    ) {
        method =
            getValueFromEnum(partnerOrderRequest.httpMethod, HttpMethod) ??
            HttpMethod.get;

        if (method === HttpMethod.get && size(partnerOrderRequest.params)) {
            href = updateQuery(
                partnerOrderRequest.url,
                partnerOrderRequest.params,
            );
        } else {
            href = partnerOrderRequest.url;
            POSTParams = partnerOrderRequest.params;
        }
    } else {
        href = deepUrl || orderUrl || null;
    }

    if (href) {
        href = applyUtm(href, seoQueryParams, clientId, medium);
    }

    return {
        href: href ?? undefined,
        method,
        POSTParams,
    };
}
