import {TransportType} from '../transportType';
import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import ISubSegment from '../../interfaces/segment/ISubSegment';
import ISegment from '../../interfaces/segment/ISegment';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import IStateSeoQueryParams from '../../interfaces/state/IStateSeoQueryParams';
import IStateFlags from '../../interfaces/state/flags/IStateFlags';

import applyUtm from '../url/applyUtm';
import getThreadUrl from '../url/getThreadUrl';
import isTransfer from './isTransfer';
import getPlaneThreadUrl from '../url/getPlaneThreadUrl';
import isSegment from './isSegment';

interface IGetThreadUrlForSegmentParams {
    segment: ISegment | ISubSegment | ITransfer | ITransferSegment;
    tld: Tld;
    language: Lang;

    seoQueryParams?: IStateSeoQueryParams;
    clientId?: string;
    utmMedium?: string;
    flags?: IStateFlags;
    isProduction?: boolean;
    isToCitySearchContext: boolean;
}

// Вернет ссылку на страницу нитки для сегмента
export default function getThreadUrlForSegment({
    segment,
    tld,
    language,

    isToCitySearchContext,
    seoQueryParams,
    clientId,
    utmMedium,
    flags,
    isProduction,
}: IGetThreadUrlForSegmentParams): string | undefined {
    if (isTransfer(segment)) {
        return undefined;
    }

    const intervalThreadDepartureFromDate =
        isSegment(segment) && segment.isInterval
            ? segment.intervalThreadDepartureFromDate
            : undefined;

    const {
        transport: {code: transportType},
        thread,
        stationFrom,
        stationTo,
        departure,
        departureMoment,
    } = segment;

    if (isSegment(segment) && segment.isArchival) {
        return undefined;
    }

    if (transportType === TransportType.plane) {
        let planeUrl = '';
        const isDynamic = 'isDynamic' in segment ? segment.isDynamic : false;

        // Явно указана ссылка на нитку
        if ('url' in segment && segment.url) {
            planeUrl = segment.url;
        }

        // Формирование ссылки по номеру рейса
        if (!planeUrl && thread && !isDynamic) {
            planeUrl = getPlaneThreadUrl({
                numberPlane: thread.number,
                departure,
                departureMoment,
                isProduction,
            });
        }

        if (planeUrl) {
            return applyUtm(planeUrl, seoQueryParams, clientId, utmMedium);
        }
    }

    if (!thread) {
        return undefined;
    }

    let stationToId;
    let stationFromId;

    if (isSegment(segment)) {
        stationToId = segment.stationTo.idFromSearchApi ?? stationTo.id;
        stationFromId = segment.stationFrom.idFromSearchApi ?? stationFrom.id;
    } else {
        stationToId = stationTo.id;
        stationFromId = stationFrom.id;
    }

    const isCanonicalurl = !flags?.__notCanonicalThreadUid;

    return getThreadUrl({
        tld,
        language,
        toCity: isToCitySearchContext,
        threadUid: thread.uid,
        canonicalUid:
            thread.canonicalUid && isCanonicalurl
                ? thread.canonicalUid
                : undefined,
        stationFromId,
        stationToId,
        departureFromMoment: departureMoment,
        departureFromDateRobot: intervalThreadDepartureFromDate,
        departure,
    });
}
