import {getCommonMask, replaceMaskValue} from '../date/mask';
import {SUBURBAN_TYPE, PLANE_TYPE} from '../transportType';
import {CHAR_NBSP, CHAR_EM_DASH} from '../stringUtils';
import {getObjectWithoutProps} from '../objectUtils';

function isAvailableForMerge({isTransfer, isInterval, transport}) {
    return transport.code !== SUBURBAN_TYPE && !(isTransfer || isInterval);
}

function getLiteHash(segment) {
    return [
        segment.title,
        segment.number,
        segment.stops,
        segment.stationFrom.id,
        segment.stationTo.id,
        segment.transport.code,
    ].join();
}

function getExtendedHash(segment) {
    return [
        segment.departure && segment.departure.substr(10),
        segment.arrival && segment.arrival.substr(10),
        segment.company && segment.company.id,
        segment.isThroughTrain,
        JSON.stringify(getObjectWithoutProps(segment.thread, ['uid'])),
        segment.stationFrom.platform,
        segment.stationTo.platform,
    ].join();
}

function getHash(segment, hashHandler, index) {
    return isAvailableForMerge(segment) ? hashHandler(segment) : index;
}

function mergeDuplicates(segments) {
    const baseSegment = segments[0];

    baseSegment.isMerged = segments.length > 1;

    if (baseSegment.isMerged) {
        baseSegment.runDays = getCommonMask(
            segments.map(({runDays = {}, thread = {}}) => ({
                runDays: replaceMaskValue(runDays, thread.uid),
            })),
        );

        const baseTransport = baseSegment.transport;
        const baseModel = baseTransport.model || {};

        baseSegment.transport.model =
            baseTransport.code !== PLANE_TYPE
                ? segments.every(
                      ({transport}) =>
                          transport.model &&
                          transport.model.title === baseModel.title,
                  )
                    ? baseTransport.model
                    : null
                : {
                      title: segments
                          .map(
                              ({transport}) =>
                                  transport.model && transport.model.title,
                          )
                          .filter(model => model)
                          .join(`${CHAR_NBSP}${CHAR_EM_DASH} `),
                  };
    }

    return baseSegment;
}

function getHashMap(segments, hashHandler) {
    return segments.reduce((map, segment, index) => {
        const hash = getHash(segment, hashHandler, index);

        if (!map[hash]) {
            map[hash] = [];
        }

        map[hash].push(segment);

        return map;
    }, {});
}

function extendHashMap(hashMap) {
    return Object.keys(hashMap).reduce((map, key) => {
        if (hashMap[key].length > 1) {
            const fullHashMap = getHashMap(hashMap[key], getExtendedHash);

            Object.keys(fullHashMap).forEach(fullKey => {
                map[fullKey] = fullHashMap[fullKey];
            });
        } else {
            map[key] = hashMap[key];
        }

        return map;
    }, {});
}

export default function mergeSegments(segments) {
    const hashMap = extendHashMap(getHashMap(segments, getLiteHash));

    return Object.keys(hashMap).map(key => mergeDuplicates(hashMap[key]));
}
