import moment, {Moment} from 'moment';

import Tld from '../../interfaces/Tld';
import SearchSegment from '../../interfaces/state/search/SearchSegment';
import SearchSegmentFromApi from '../../interfaces/state/search/SearchSegmentFromApi';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import ISegmentFromApi from '../../interfaces/segment/ISegmentFromApi';
import ISegment from '../../interfaces/segment/ISegment';
import ITransferFromApi from '../../interfaces/transfer/ITransferFromApi';
import ITransfer from '../../interfaces/transfer/ITransfer';
import EnvironmentType from '../../interfaces/EnvironmentType';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import SegmentEventType from '../../interfaces/segment/SegmentEventType';
import SegmentCancelType from '../../interfaces/SegmentCancelType';
import ISegmentEvent from '../../interfaces/segment/ISegmentEvent';
import ISegmentThread from '../../interfaces/segment/ISegmentThread';

import patchTitle from './patchTitle';
import patchTariffs from './tariffs/patchTariffs';
import isAllDaysSearch, {IIsAllDaysSearch} from '../search/isAllDaysSearch';
import isSegmentFromApi from './isSegmentFromApi';
import getSegmentUniqId from './getSegmentUniqId';

let counter = 0;

interface IPatchSegments<T extends SearchSegmentFromApi> {
    segments: T[];
    meta: {
        tld: Tld;
        context: {
            time: {
                now: number;
            };
            when?: {
                special?: DateSpecialValue;
            };
        };
        environment: EnvironmentType;
        currencies: IStateCurrencies;
        isProduction: boolean;
    };

    isDynamic?: boolean;
}

export function isGoneCheck(
    context: IIsAllDaysSearch,
    now: Moment,
    departure: string,
): boolean {
    return !isAllDaysSearch(context) && now > moment(departure);
}

function getCancelType(
    arrivalEvent: ISegmentEvent | null,
    departureEvent: ISegmentEvent | null,
    thread: ISegmentThread | null,
): SegmentCancelType | null {
    const arrivalStationCanceled = Boolean(
        arrivalEvent && arrivalEvent.type === SegmentEventType.cancelled,
    );
    const departureStationCanceled = Boolean(
        departureEvent && departureEvent.type === SegmentEventType.cancelled,
    );

    if (arrivalStationCanceled && departureStationCanceled) {
        return SegmentCancelType.full;
    }

    if (arrivalStationCanceled || departureStationCanceled) {
        return SegmentCancelType.partial;
    }

    if (thread) {
        if (thread.cancelled) {
            return SegmentCancelType.full;
        }

        // thread.cancelled может быть false, но это значит что отмены есть, просто частичные
        if (thread.cancelled === false) {
            return SegmentCancelType.partial;
        }
    }

    return null;
}

export default function patchSegments(
    params: IPatchSegments<ISegmentFromApi>,
): ISegment[];
export default function patchSegments(
    params: IPatchSegments<ITransferFromApi>,
): ITransfer[];
export default function patchSegments(
    params: IPatchSegments<SearchSegmentFromApi>,
): SearchSegment[];
export default function patchSegments({
    segments,
    meta,
    isDynamic = false,
}: IPatchSegments<SearchSegmentFromApi>): SearchSegment[] {
    const {context, environment} = meta;
    const {
        time: {now: nowTimeInMillisecond},
    } = context;
    const now = moment(nowTimeInMillisecond);

    return segments.map(segment => {
        const isGone = isGoneCheck(context, now, segment.departure);
        const timezoneFrom = segment.stationFrom.timezone;
        const timezoneTo = segment.stationTo.timezone;
        const segmentId = `${environment}-${counter++}`;

        if (isSegmentFromApi(segment)) {
            const {subSegments, isThroughTrain, arrivalEvent, departureEvent} =
                segment;

            return {
                id: getSegmentUniqId(segment),
                ...segment,
                isDynamic,
                isGone,
                timezoneFrom,
                timezoneTo,
                segmentId,
                trainPartners: [], // будут добавлены в редьюсере
                badges: {},
                thread: segment.thread || undefined,
                isThroughTrain: Boolean(isThroughTrain),
                subSegments:
                    subSegments &&
                    subSegments.map(subSegment => ({
                        timezoneFrom: subSegment.stationFrom.timezone,
                        timezoneTo: subSegment.stationTo.timezone,
                        ...subSegment,
                    })),
                tariffsKeys:
                    segment.tariffsKeys ||
                    (segment.key && [segment.key]) ||
                    segment.keys ||
                    [],
                tariffClassKeys: [],
                title: patchTitle(segment.title),
                tariffs: segment.tariffs
                    ? patchTariffs(segment, segment.tariffs, meta)
                    : undefined,
                cancelType: getCancelType(
                    arrivalEvent,
                    departureEvent,
                    segment.thread || null,
                ),
            };
        }

        // пересадка
        return {
            ...segment,
            isDynamic,
            isGone,
            timezoneFrom,
            timezoneTo,
            segmentId,
            badges: {},
            segments: segment.segments.map(transferSegment => ({
                ...transferSegment,
                timezoneFrom: transferSegment.stationFrom.timezone,
                timezoneTo: transferSegment.stationTo.timezone,
                isGone: isGoneCheck(context, now, transferSegment.departure),
                tariffClassKeys: [],
                tariffs: transferSegment.tariffs
                    ? patchTariffs(
                          transferSegment,
                          transferSegment.tariffs,
                          meta,
                      )
                    : undefined,
                badges: undefined,
            })),
            tariffClassKeys: [],
            tariffs: segment.tariffs
                ? patchTariffs(segment, segment.tariffs, meta)
                : undefined,
        };
    });
}
