import qs from 'qs';

import Tld from '../../../interfaces/Tld';
import {OrderUrlOwner} from '../tariffClasses';
import IStateCurrencies from '../../../interfaces/state/IStateCurrencies';
import ISegmentTariffs from '../../../interfaces/segment/ISegmentTariffs';
import ISegmentTariffClass from '../../../interfaces/segment/ISegmentTariffClass';
import ISegmentTariffClassFromBackend from '../../../interfaces/segment/ISegmentTariffClassFromBackend';
import ISegmentFromApi from '../../../interfaces/segment/ISegmentFromApi';
import ITransferSegmentFromApi from '../../../interfaces/transfer/ITransferSegmentFromApi';
import ITransferFromApi from '../../../interfaces/transfer/ITransferFromApi';
import ISegmentCompany from '../../../interfaces/segment/ISegmentCompany';

import convertPrice from '../../currencies/convertPrice';
import isAllDaysSearch, {IIsAllDaysSearch} from '../../search/isAllDaysSearch';
import areTrainTariffsAvailableForSegment from './areTrainTariffsAvailableForSegment';
import getTrainUrl from '../../url/getTrainUrl';
import isTransferFromApi from '../isTransferFromApi';

interface IMetaForPatchTariffs {
    tld: Tld;
    context: IIsAllDaysSearch;
    isProduction: boolean;
    currencies: IStateCurrencies;
}

/*
 * Метод конвертирует цену в национальную валюту
 *  - это необходимо для корректной сортировки по цене, т.к. возможен случай когда тарифы приходят в разных валютах
 * Метод формирует урл для новой покупки
 *  - формируем на этапе патчинга, чтоб не прокидывать кучу параметров сквозь компоненты
 */
export default function patchTariffs(
    segment: ISegmentFromApi | ITransferFromApi | ITransferSegmentFromApi,
    tariffs: ISegmentTariffs<ISegmentTariffClassFromBackend>,
    meta: IMetaForPatchTariffs,
    codeshareCompany?: ISegmentCompany,
    codeshareNumber?: string,
): ISegmentTariffs<ISegmentTariffClass> {
    const {classes} = tariffs;
    const {tld, isProduction, context, currencies} = meta;

    const nationalPreferred = {
        ...currencies,
        preferredCurrency: currencies.nationalCurrency,
    };
    const allDaysSearch = isAllDaysSearch(context);

    return {
        ...tariffs,
        classes: Object.keys(classes).reduce((updatedClasses, key) => {
            const tariffClass = classes[key];
            const {price, trainOrderUrl, trainOrderUrlOwner} = tariffClass;

            updatedClasses[key] = {
                ...tariffClass,
                nationalPrice: convertPrice(price, nationalPreferred),
                codeshareCompany,
                codeshareNumber,
            };

            if (
                trainOrderUrl &&
                !allDaysSearch && // для поиска на все дни ссылки формируются динамически в OrderCalendarDayLink
                !isTransferFromApi(segment) &&
                areTrainTariffsAvailableForSegment(segment)
            ) {
                let orderUrl = '';

                switch (trainOrderUrlOwner) {
                    case OrderUrlOwner.trains: {
                        const [path, search = ''] = trainOrderUrl.split('?');
                        const query = qs.parse(search);

                        orderUrl = getTrainUrl({
                            isProduction,
                            path,
                            query: {...query, citizenship: tld},
                        });
                        break;
                    }

                    case OrderUrlOwner.ufs: {
                        orderUrl = trainOrderUrl;
                        break;
                    }
                }

                updatedClasses[key].orderUrl = orderUrl;
            }

            return updatedClasses;
        }, {}),
    };
}
