import {SUBURBAN_TYPE, ALL_TYPE} from '../../transportType';
import {CHAR_EM_DASH, CHAR_NBSP} from '../../stringUtils';

// eslint-disable-next-line no-duplicate-imports
import {humanList} from '../../stringUtils';

import segmentTransferKeyset from '../../../i18n/segment-transfer';

function getStationTitle({title, popularTitle, settlement}) {
    let stationTitle = popularTitle || title;

    if (settlement && !stationTitle.includes(settlement.title)) {
        stationTitle += ` (${settlement.title})`;
    }

    return stationTitle;
}

/**
 * Отдаст заголовок пересадочной точки в зависимости от наличия предложного падежа
 * - Если предложный падеж есть, вернет фраза: "с пересадкой в Москве"
 * - Если предложного падежа нет, вернет фразу "пересадка: Москва"
 *
 * @param {Object} transferPoint
 * @param {Function} keyset
 *
 * @return {string}
 */
function getSingleTransferTitle(transferPoint, keyset) {
    if (transferPoint.titleLocative && transferPoint.preposition) {
        return keyset('transfer-title-single-locative', {
            transfer: transferPoint.titleLocative,
            preposition: transferPoint.preposition,
        });
    }

    return keyset('transfer-title-single-infinitive', {
        transfer: transferPoint.title,
    });
}

/**
 * Отдаст заголовок пересадочных точек в зависимости от наличия винительного падежа у всех точек
 * - Если винительный падеж есть у всех точек, вернет фразу "через Москву, Владимир и Муром"
 * - Если винительного падежа нет у какой-либо точки, вернет фразу "пересадки: Москва, Владимир, Муром"
 *
 * @param {Object[]} transferPoints
 * @param {Function} keyset
 * @param {string} language
 *
 * @return {string}
 */
function getMultipleTransferTitle(transferPoints, keyset, language) {
    const pointTitlesAccusative = transferPoints.map(
        point => point.titleAccusative,
    );

    if (pointTitlesAccusative.every(title => Boolean(title))) {
        return keyset('transfer-title-multiple-through', {
            transfers: humanList(pointTitlesAccusative, language),
        });
    }

    const pointTitlesInfinitive = transferPoints.map(point => point.title);

    return keyset('transfer-title-multiple-infinitive', {
        transfers: pointTitlesInfinitive.join(', '),
    });
}

function getTransferTitle(transferPoints, language) {
    if (transferPoints.length === 0) {
        return null;
    }

    if (transferPoints.length === 1) {
        return getSingleTransferTitle(transferPoints[0], segmentTransferKeyset);
    }

    return getMultipleTransferTitle(
        transferPoints,
        segmentTransferKeyset,
        language,
    );
}

/**
 * Патчит тип транспорта.
 * Возвращает "электричку", если искали "любым" транспортом, и все сегменты внутри пересадочного сегмента - электрички.
 *
 * @param {Object} segment
 * @param {string} transportType
 *
 * @return {string}
 */
function patchTransportType(segment, transportType) {
    if (transportType === ALL_TYPE) {
        if (
            segment.segments.every(
                subsegment => subsegment.transport.code === SUBURBAN_TYPE,
            )
        ) {
            return SUBURBAN_TYPE;
        }
    }

    return transportType;
}

/**
 * Агрегация данных одним проходом для вывода в снипетах пересадочника
 *
 * Логика: Под временем сниппета нужно выводить полный список станций (transferStations)
 *         В заголовке сниппета нужно выводить города пересадок (transferTitle), или если это электрички,
 *         то станции пересадок, однако, если при пересадке электричек нужно бежать на другую станцию,
 *         то в заголовке выводить город вместо двух станций
 *
 * @param {Object} transfer - сегмент пересадочника
 * @param {string} language
 * @param {string} transportType - тип транспорта из контекста поиска
 * @return {Object} например { transferStations: "Екатеринбург-Пасс. - Шаля - Пермь-2"], transferTitle: "Пересадка в Шале" transferPoints: ["Шаля"] }
 */
export default function getAggregatedInfo({transfer, language, transportType}) {
    const segmentTransportType = patchTransportType(transfer, transportType);
    const lastSegmentIndex = transfer.segments.length - 1;
    const {transferStations, transferPoints} = transfer.segments.reduce(
        (value, currentSegment, index, arr) => {
            const prevSegment = arr[index - 1];
            const nextSegment = arr[index + 1];

            // Если это первый сегмент или у нас пересадка на разных станциях в пределах одного города
            // Прим.: прилетели в Москву и надо пересадиться из Шереметьево в Домодедово ( пример из вакуумного коня)
            if (
                index === 0 ||
                prevSegment.stationTo.id !== currentSegment.stationFrom.id
            ) {
                value.transferStations.push(currentSegment.stationFrom);
            }

            value.transferStations.push(currentSegment.stationTo);

            if (index < lastSegmentIndex) {
                if (
                    !currentSegment.stationTo.settlement ||
                    (segmentTransportType === SUBURBAN_TYPE &&
                        currentSegment.stationTo.id ===
                            nextSegment.stationFrom.id)
                ) {
                    value.transferPoints.push(currentSegment.stationTo);
                } else {
                    value.transferPoints.push(
                        currentSegment.stationTo.settlement,
                    );
                }
            }

            return value;
        },
        {
            transferStations: [],
            transferPoints: [],
        },
    );

    return {
        transferStations: transferStations
            .map(station => getStationTitle(station))
            .join(`${CHAR_NBSP}${CHAR_EM_DASH} `),
        transferTitle: getTransferTitle(transferPoints, language),
    };
}
