import {
    MODEL,
    DENSITY,
    ETICKET,
    TROUGH_TRAIN,
    DELUXE_TRAIN,
    DYNAMIC_PRICING,
    ENDPOINT_EVENT_DATA,
} from './features';

import getSubtype from './getSubtype';
import getCompanies from './getCompanies';
import getEndpointsData from '../endpointEvents/getEndpointsData';
import hasElectronicRegistration from './hasElectronicRegistration';

import keyset from '../../../i18n/segment';

/**
 * Метод получения информации о транспорте
 * @param {Object} segment - данные сегмента
 * @param {boolean} isMobile - моблильное ли устройство
 * @param {string} tld
 * @param {string} language
 * @param {boolean | undefined} isSuburbanSearchResult - поиск электричками или нет
 * @return {Array}
 */
export default function getTransportFeatures(
    segment,
    isMobile,
    tld,
    language,
    isSuburbanSearchResult,
) {
    const {
        thread,
        transport,
        isThroughTrain,
        company,
        companies,
        tariffs,
        hasDynamicPricing,
    } = segment;

    const features = [];
    const {model, code: transportType} = transport;
    const {deluxeTrain, density} = thread || {};

    const endpointsData = getEndpointsData(segment);
    const deluxeTrainName = deluxeTrain && deluxeTrain.shortTitle;
    const subtype = getSubtype(transport, thread, isSuburbanSearchResult);

    if (endpointsData && !endpointsData.isGone) {
        features.push({
            type: ENDPOINT_EVENT_DATA,
            props: {...endpointsData},
        });
    }

    if (
        hasElectronicRegistration(
            transportType,
            tariffs && tariffs.electronicTicket,
        )
    ) {
        features.push({
            type: ETICKET,
        });
    }

    if (hasDynamicPricing) {
        features.push({
            type: DYNAMIC_PRICING,
        });
    }

    switch (true) {
        case Boolean(subtype):
            features.push(subtype);
            break;
        case isThroughTrain:
            features.push({
                type: TROUGH_TRAIN,
                content: keyset('through-train'),
            });
            break;
        case Boolean(deluxeTrainName): {
            const feature = {
                type: DELUXE_TRAIN,
                content: deluxeTrainName,
            };

            const deluxeTrainHref = deluxeTrain && deluxeTrain.pagePath;

            if (deluxeTrainHref) {
                feature.props = {href: deluxeTrainHref};
            }

            features.push(feature);

            break;
        }
    }

    features.push(...getCompanies(tld, language, company, companies));

    if (model && model.title) {
        features.push({
            type: MODEL,
            content: model.title,
        });
    }

    if (!isMobile && density) {
        features.push({
            type: DENSITY,
            content: thread.density,
        });
    }

    return features;
}
