import {YBUS} from './tariffSources';

import {TransportType} from '../transportType';
import ISegment from '../../interfaces/segment/ISegment';
import ISegmentFromApi from '../../interfaces/segment/ISegmentFromApi';
import ISearchMeta from '../../interfaces/state/search/ISearchMeta';

import patchTariffs from './tariffs/patchTariffs';
import patchSegments from './patchSegments';
import getBestSegment from './getBestSegment';
import {addVariantsToBusSegment} from './addVariantsToBusSegment';
import updateYBusSegment from './buses/updateYBusSegment';

// Вернёт номер сегмента отображаемый в сниппете
function getMergedNumber(
    baseSegment: ISegment, // базовый сегмент из ответа ручки search
    bestSegment: ISegmentFromApi, // сегмент с ценами (из ручек tariffs, train-tariffs, bus-tariffs)
    bestSegmentIsCodeshare: boolean,
): string {
    const {
        isMetaSegment,
        hasTrainTariffs,
        number: baseNumber,
        transport: {code: transportType},
    } = baseSegment;

    if (bestSegmentIsCodeshare) {
        return baseNumber;
    }

    const isSuburbanWithTrainTariffs =
        transportType === TransportType.suburban && hasTrainTariffs;
    const shouldUseOriginNumber = isMetaSegment || isSuburbanWithTrainTariffs;

    return shouldUseOriginNumber
        ? baseNumber
        : bestSegment.number || baseNumber;
}

function getMergeUrl(
    baseSegment: ISegment,
    bestSegment: ISegmentFromApi,
    bestSegmentIsCodeshare: boolean,
    updateSegments: ISegmentFromApi[],
): string | undefined {
    if (!bestSegmentIsCodeshare) {
        return bestSegment.url || baseSegment.url;
    }

    if (baseSegment.url) {
        return baseSegment.url;
    }

    // Попробуем поискать url базового сегмента в сегментах с тарифами
    const tariffSegmentForBase = updateSegments.find(
        segment => segment.number === baseSegment.number && segment.url,
    );

    return tariffSegmentForBase && tariffSegmentForBase.url;
}

// Метод добавляет информацию о тарифах к базовому сегменту
export default function updateSegment(
    baseSegment: ISegment, // базовый сегмент
    updateSegments: ISegmentFromApi[], // массив сегментов с ценами (из ручек tariffs, train-tariffs, bus-tariffs)
    meta: ISearchMeta,
): ISegment {
    // Если время отправления базового сегмента и сегмента с ценой отличается,
    // то заменяем базовый сегмент на сегмент из тарифов (сегмент пополнение).
    // Предполагается, что сегмент пополнение содержит более актуальную информацию.
    for (let i = 0; i < updateSegments.length; i++) {
        const segment = updateSegments[i];

        if (baseSegment.departure !== segment.departure) {
            return patchSegments({
                segments: [segment],
                meta,
                isDynamic: true,
            })[0];
        }
    }

    let bestSegment = getBestSegment(baseSegment, updateSegments);

    // Определяем является ли лучший сегмент кодшером базового
    const codeshare = baseSegment.codeshares?.find(
        ({number}) => bestSegment.number === number,
    );
    const codeshareCompany = codeshare?.company;
    const bestSegmentIsCodeshare = Boolean(codeshareCompany);

    // Добавляем варианты для выбора партнера к лучшему сегменту, если это автобусы
    if (baseSegment.transport.code === TransportType.bus) {
        if (bestSegment === baseSegment) {
            const secondBestSegment = getBestSegment(
                updateSegments[0],
                updateSegments,
            );

            bestSegment = addVariantsToBusSegment(
                bestSegment,
                secondBestSegment,
                false,
            );
        } else {
            bestSegment = addVariantsToBusSegment(bestSegment, baseSegment);
        }
    } else if (bestSegment === baseSegment) {
        // Если к базовому сегменту уже добавлена цена и цена ниже чем в updateSegments,
        // то возвращаем базовый сегмент.
        return baseSegment;
    }

    if (bestSegment.source === YBUS) {
        // Если есть несколько вариантов, не обрабатываем сегмент как сегмент Я.Автобусов
        if (bestSegment.hasVariants) {
            bestSegment.source = undefined;
        } else {
            return updateYBusSegment(baseSegment, bestSegment, meta);
        }
    }

    // начальная дата нужна для формирования правильной ссылки в patchTariffs
    if (!bestSegment.startDate) {
        bestSegment.startDate = baseSegment.startDate;
    }

    // Иногда ручка тарифов не знает конечную станцию и присылает вместо id станции id города.
    // (Также возможно, что вместо id придет id другой станции (не переданной в ручку партнеров))
    // Ранее из ручки search нам уже приходил id конечной станции и здесь мы его сохраняем в отдельное поле,
    //      чтобы формировать правильную ссылку на страницу нитки.
    // Если же id совпадают, то нужно сохранить некоторые поля станций из baseSegment (pageType, mainSubtype).
    // baseSegment - сегмент с данными из ручки search.
    // bestSegment - сегмент с данными от ручек партнеров.
    const stationTo =
        baseSegment.stationTo.id === bestSegment.stationTo.id
            ? {...baseSegment.stationTo, ...bestSegment.stationTo}
            : {
                  ...bestSegment.stationTo,
                  idFromSearchApi: baseSegment.stationTo.id,
              };

    // Аналогичная логика, как и для конечной станции
    const stationFrom =
        baseSegment.stationFrom.id === bestSegment.stationFrom.id
            ? {...baseSegment.stationFrom, ...bestSegment.stationFrom}
            : {
                  ...bestSegment.stationFrom,
                  idFromSearchApi: baseSegment.stationFrom.id,
              };

    // Дополняем базовый сегмент тарифами и данными, которые содержатся только
    // в ответе ручек тарифов.
    return {
        ...baseSegment,
        stationTo,
        stationFrom,
        originalNumber: bestSegmentIsCodeshare
            ? baseSegment.originalNumber
            : bestSegment.originalNumber,
        number: getMergedNumber(
            baseSegment,
            bestSegment,
            bestSegmentIsCodeshare,
        ),
        hasDynamicPricing: bestSegment.hasDynamicPricing,
        hasVariants: bestSegment.hasVariants,
        company: bestSegmentIsCodeshare
            ? baseSegment.company || bestSegment.company
            : bestSegment.company || baseSegment.company,
        tariffs:
            bestSegment.tariffs &&
            patchTariffs(
                bestSegment,
                bestSegment.tariffs,
                meta,
                codeshareCompany,
                codeshare?.number,
            ),
        url: getMergeUrl(
            baseSegment,
            bestSegment,
            bestSegmentIsCodeshare,
            updateSegments,
        ),
        useCityInsteadStationTo: bestSegment.useCityInsteadStationTo,
        useCityInsteadStationFrom: bestSegment.useCityInsteadStationFrom,
    };
}
