import ISegment from '../../interfaces/segment/ISegment';
import ISegmentFromApi from '../../interfaces/segment/ISegmentFromApi';
import ISearchMeta from '../../interfaces/state/search/ISearchMeta';

import updateSegment from './updateSegment';
import getTariffMatchWeigh from './getTariffMatchWeigh';
import patchSegments from './patchSegments';
import clearDuplicateTrainTariffs from './clearDuplicateTrainTariffs';

// Склейка базовых сегментов с тарифами
export default function updateSegments(
    baseSegments: ISegment[], //  массив базовых сегментов (из ответа ручки search)
    tariffSegments: ISegmentFromApi[], // массив сегментов с тарифами (из ответов ручек tariffs, train-tariffs, bus-tariffs)
    meta: ISearchMeta, // метаданные из stor'а
): ISegment[] {
    const {context} = meta || {};
    const {latestDatetime} = context || {};
    const matchMap: Record<number, ISegmentFromApi[]> = {};
    const unmatchedTariffSegments: ISegmentFromApi[] = [];

    // Фильтруем сегменты тарифов:
    // если сегменты были запрошены на двое суток - отбрасываем те что позже 4х часов последних суток
    // если сегмент тарифов может расширить более одного базового сегмента - отбрасываем его и логируем
    for (let i = 0; i < tariffSegments.length; i++) {
        const tariffSegment = tariffSegments[i];

        if (latestDatetime && tariffSegment.departure > latestDatetime) {
            continue;
        }

        const matched: {
            baseSegment: ISegment;
            matchWeight: number;
            matchIndex: number;
        }[] = [];

        for (let j = 0; j < baseSegments.length; j++) {
            const baseSegment = baseSegments[j];

            const matchWeight = getTariffMatchWeigh(baseSegment, tariffSegment);

            if (matchWeight > 0) {
                matched.push({baseSegment, matchWeight, matchIndex: j});

                if (!matchMap[j]) {
                    matchMap[j] = [];
                }

                matchMap[j].push(tariffSegment);
            }
        }

        if (!matched.length) {
            // canSupplySegments = false означает, что сегмент может только дополнять существующие сегменты,
            // отображать его как рейс пополнение нельзя
            if (tariffSegment.canSupplySegments !== false) {
                unmatchedTariffSegments.push(tariffSegment);
            }
        } else if (matched.length > 1) {
            const matchedSorted = matched.sort(
                ({matchWeight: a}, {matchWeight: b}) => b - a,
            );

            if (matchedSorted[0].matchWeight === matchedSorted[1].matchWeight) {
                tariffSegment.ambiguous = true;
                matched.forEach(({baseSegment: item}) => {
                    if (!item.missedTariffs) {
                        item.missedTariffs = [];
                    }

                    if (
                        tariffSegment.yBusId &&
                        !item.missedTariffs.includes(tariffSegment.yBusId)
                    ) {
                        item.missedTariffs.push(tariffSegment.yBusId);
                    }
                });
            } else {
                matchedSorted.slice(1).forEach(({matchIndex}) => {
                    const item = matchMap[matchIndex];
                    const spliceIndex = item.indexOf(tariffSegment);

                    item.splice(spliceIndex, 1);
                });
            }
        }
    }

    // расширяем сегменты тарифами
    const updatedSegments: ISegment[] = [];

    for (let i = 0; i < baseSegments.length; i++) {
        // при расширении сегмента ценами исключаем из массива тарифов неоднозначные элементы
        const matched =
            matchMap[i] && matchMap[i].filter(item => !item.ambiguous);

        updatedSegments.push(
            matched && matched.length
                ? updateSegment(baseSegments[i], matched, meta)
                : baseSegments[i],
        );
    }

    const segments = [
        ...updatedSegments,
        ...patchSegments({
            segments: unmatchedTariffSegments,
            meta,
            isDynamic: true,
        }),
    ];

    return clearDuplicateTrainTariffs(segments);
}
