import {momentTimezone as moment} from '../../../reexports';

import {buildTransportForms} from '../transportType';
import {humanList} from '../stringUtils';

import isResultEmpty from '../search/isResultEmpty';
import isAllDaysSearch from '../search/isAllDaysSearch';
import isTodaySearch from '../search/isTodaySearch';

import {humanizeDuration} from '../date/formats';

import formatPrice from '../format/formatPrice';

import transportFilter from '../filters/transport';

import SegmentSubtypeCode from '../../interfaces/segment/SegmentSubtypeCode';
// eslint-disable-next-line no-duplicate-imports
import {FilterTransportType} from '../transportType';

import {getMinPricesByClass} from './getMinPricesByClass';
import {getSubtypesWithPrices} from './getSubtypesWithPrices';
import getMinDurationSegment from '../segments/getMinDurationSegment';
import getSearchSubtypes from '../segments/transportFeatures/getSearchSubtypes';
import gatherMinPriceData from '../segments/gatherMinPriceData';
import groupFilteredSegments from '../segments/groupFilteredSegments';
import isLastochkaPage from '../search/isLastochkaPage';
import {isForeignToForeignSearch} from '../search/isForeignToForeignSearch';

import insertMarkupIntoKey from '../../i18n/_utils/insertMarkupIntoKey';

import descriptionKeyset from '../../i18n/search-description-v2';

export function shouldShowSearchDescription(search, page) {
    const {context, segments, archivalData} = search;

    return (
        Boolean(archivalData) ||
        (!isResultEmpty({segments, page}) &&
            (isAllDaysSearch(context) || isTodaySearch(context)))
    );
}

/**
 * Получает года для показа для определенного момента времени
 * @param {Object} time
 * @return {number[]}
 */
export function getYears(time) {
    const {now, timezone} = time;
    const nowMoment = moment.tz(now, timezone);
    const year = nowMoment.year();

    return nowMoment.month() < 10 ? [year] : [year, year + 1];
}

/**
 * Возвращает сегменты, сгруппированные по виду транспорта
 * @param {Array} segments
 * @param {Object} context
 * @return {Array} - массив объектов формата {title: {string}, segments: {Array}}
 */
const groupSegmentsByTransport = (segments, context) => {
    return groupFilteredSegments({
        filterManager: transportFilter,
        segments,
        context,
    });
};

export function buildSearchDescription({
    context,
    language,
    segments,
    transportTypes,
    filtering,
    minPriceData,
    archivalData,
}) {
    if (!minPriceData) {
        minPriceData = {};
    }

    const transportForms = buildTransportForms(transportTypes);

    const {
        from,
        to,
        time,
        transportType: transportTypeFromContext,
        when,
    } = context;
    let transportType = transportTypeFromContext;

    if (archivalData) {
        transportType =
            archivalData.canonical.transportType || FilterTransportType.all;
    }

    if (
        transportType === FilterTransportType.all &&
        transportTypes.length === 1
    ) {
        transportType = transportTypes[0];
    }

    const minDurationSegment = getMinDurationSegment(segments);
    const searchSubtypes = getSearchSubtypes(segments, context);
    const segmentsByTransport =
        transportType === FilterTransportType.all
            ? groupSegmentsByTransport(segments, context)
            : [];
    const minPricesByClass =
        transportType === FilterTransportType.train
            ? getMinPricesByClass(segments)
            : [];
    const subtypesWithPrices = getSubtypesWithPrices(searchSubtypes);

    const params = {
        from,
        to,

        transportTypes: '',

        price: '',
        class: '',
        cheapestTransport: '',

        duration: humanizeDuration(
            moment.duration(
                minDurationSegment && minDurationSegment.duration,
                'seconds',
            ),
        ),
        year: humanList(getYears(time), language),
        subtypesWithPrices,
        isForeignSearch: isForeignToForeignSearch(from, to),
        subtypesCount: searchSubtypes ? searchSubtypes.length : 0,
    };

    if (transportType === FilterTransportType.all) {
        params.transportTypes = humanList(
            transportTypes.map(tType =>
                transportForms[tType].plural.nominative.toLowerCase(),
            ),
            language,
        );
        // Добавляем все виды транспорта и минимальное время для каждого из них
        const typesWithDuration = [];

        segmentsByTransport.forEach(transportSegmentsObject => {
            const {title: transportTitle, segments: transportSegments} =
                transportSegmentsObject;
            const typeMinDurationSegment =
                getMinDurationSegment(transportSegments);

            if (typeMinDurationSegment) {
                const {duration: typeDuration} = typeMinDurationSegment;

                const typeText = descriptionKeyset('element-from-value', {
                    title: transportTitle,
                    value: humanizeDuration(
                        moment.duration(typeDuration, 'seconds'),
                    ),
                });

                typesWithDuration.push(typeText);
            }
        });
        params.typesWithDuration = typesWithDuration.join(', ');
    }

    const currentIsLastochkaPage = isLastochkaPage(
        transportTypeFromContext,
        when,
        filtering,
    );
    const titleKey = currentIsLastochkaPage
        ? 'title-lastochka'
        : `title-${transportType}`;

    if (archivalData) {
        return {title: descriptionKeyset(titleKey, params), text: ''};
    }

    let textKey = currentIsLastochkaPage
        ? 'text-lastochka'
        : `text-${transportType}`;

    if (minPriceData.price) {
        textKey += '-price';

        switch (transportType) {
            case FilterTransportType.train: {
                // Для поездов добавляем цены на все классы
                const trainClassesWithPrices = [];

                minPricesByClass.forEach(classWithPrice => {
                    const {tariffClass, price: classPrice} = classWithPrice;

                    const classText = descriptionKeyset('element-from-value', {
                        title: descriptionKeyset(`class-${tariffClass}`),
                        value: formatPrice(classPrice, {language, round: true}),
                    });

                    trainClassesWithPrices.push(classText);
                });
                params.trainClassesWithPrices =
                    trainClassesWithPrices.join(', ');
                break;
            }

            case FilterTransportType.all: {
                // Для всех видов транспорта добавляем цены на каждый из них
                const typesWithPrices = [];

                segmentsByTransport.forEach(transportSegmentsObject => {
                    const {segments: transportSegments, key: transportKey} =
                        transportSegmentsObject;
                    const typeMinPrice = gatherMinPriceData(transportSegments);

                    if (typeMinPrice) {
                        const {price: typePrice} = typeMinPrice;

                        const typeText = descriptionKeyset(
                            'element-from-value',
                            {
                                title: descriptionKeyset(
                                    `type-${transportKey}`,
                                ),
                                value: formatPrice(typePrice, {
                                    language,
                                    round: true,
                                }),
                            },
                        );

                        typesWithPrices.push(typeText);
                    }
                });
                params.typesWithPrices = typesWithPrices.join(', ');
                break;
            }

            default:
                params.price = formatPrice(minPriceData.price, {
                    language,
                    round: true,
                });

                if (minPriceData.price.class) {
                    const classKey = `class-${minPriceData.price.class}`;

                    params.class = descriptionKeyset.get(classKey);
                }
        }
    }

    if (
        transportType === FilterTransportType.suburban &&
        !currentIsLastochkaPage
    ) {
        searchSubtypes.forEach(subtype => {
            if (subtype.key === SegmentSubtypeCode.lastochka) {
                textKey += '-lastochki';
            }
        });
    }

    if (subtypesWithPrices && transportType === FilterTransportType.train) {
        textKey += '-subtypes';
    }

    return {
        title: descriptionKeyset(titleKey, params),
        text: insertMarkupIntoKey(descriptionKeyset, textKey, params),
    };
}
