import {momentTimezone as moment} from '../../../reexports';

import {MOMENT} from '../date/formats';
import {SORTING_TYPES, SORTERS} from './utils';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';

import duration from './duration';
import price from './price';
import {time, dateTime} from './time';
import {getBaseTariffClassKeys} from '../segments/getBaseTariffClassKeys';

const sorters = {duration, price, time, dateTime};

function sortingIsAvailable({sort, queryingPrices}) {
    if (sort.by === SORTING_TYPES.PRICE) {
        return !queryingPrices;
    }

    return true;
}

function buildVector(segments, {by, reverse}) {
    const sorter = sorters[by];
    const comparator = sorter.comparator;
    const intervalComparator = sorter.intervalComparator || comparator;
    const multiplier = reverse ? -1 : 1;

    return Array.from({length: segments.length}, (item, index) => index).sort(
        (indexA, indexB) => {
            const segmentA = segments[indexA];
            const segmentB = segments[indexB];

            const aIsInterval = Boolean(segmentA.isInterval);
            const compare = aIsInterval ? intervalComparator : comparator;

            if (aIsInterval === Boolean(segmentB.isInterval)) {
                return compare(segmentA, segmentB) * multiplier;
            }

            return aIsInterval ? -1 : 1;
        },
    );
}

// Разбиваем сегменты по количеству содержащихся тарифов
function splitSegments(segments, order) {
    return segments.reduce(
        (result, segment, index) => {
            const tariffs = getBaseTariffClassKeys(segment);
            const noPrice = !tariffs.length;

            if (noPrice) {
                result.splittedSegments.push({...segment, index});
                result.sortData.push({
                    price: {value: order ? -Infinity : Infinity},
                });
            } else {
                tariffs.forEach(tariffName => {
                    const tariffItem = segment.tariffs.classes[tariffName];

                    result.sortData.push({
                        price: tariffItem.nationalPrice || {
                            value: order ? -Infinity : Infinity,
                        },
                    });
                    result.splittedSegments.push({
                        ...segment,
                        index,
                        segmentId: `${segment.segmentId}-${tariffName}`,
                        tariffs: {
                            ...segment.tariffs,
                            classes: {
                                [tariffName]: tariffItem,
                            },
                        },
                    });
                });
            }

            return result;
        },
        {
            splittedSegments: [],
            sortData: [],
        },
    );
}

// Соединяем соседние сегменты если они соответствуют одному и тому же исходному сегменту
function joinSegments(segments, vector) {
    return vector.reduce((result, item) => {
        const tail = result[result.length - 1];
        const currentSegment = segments[item];

        if (tail && tail.index === currentSegment.index) {
            tail.tariffs.classes = {
                ...tail.tariffs.classes,
                ...currentSegment.tariffs.classes,
            };
        } else {
            result.push(currentSegment);
        }

        return result;
    }, []);
}

export function updateSortingIndices(searchState) {
    const {sort, sortMapping} = searchState;

    if (sort.by === SORTING_TYPES.PRICE) {
        if (sortingIsAvailable(searchState)) {
            const {segments = []} = sortMapping;
            const {filteredSegmentIndices} = searchState.filtering;

            return {
                ...sortMapping,
                filteredSegmentIndices: segments.map(
                    item => filteredSegmentIndices[item.index],
                ),
            };
        }
    }

    return sortMapping;
}

export function sortSegments(segments, searchState) {
    const {sort, context} = searchState;
    const {by, reverse} = sort;

    switch (by) {
        case SORTING_TYPES.PRICE:
            if (sortingIsAvailable(searchState)) {
                // Сортировка по цене разбивает сегменты, у которых несколько цен на самостоятельные сегменты с одной ценой
                const {splittedSegments, sortData} = splitSegments(
                    segments,
                    sort.reverse,
                );
                const vector = buildVector(sortData, sort);
                const joinedSegments = joinSegments(splittedSegments, vector);
                const {filteredSegmentIndices} = searchState.filtering;

                return {
                    vector: Array.from(
                        {length: joinedSegments.length},
                        (item, index) => index,
                    ),
                    segments: joinedSegments,
                    filteredSegmentIndices: joinedSegments.map(
                        item => filteredSegmentIndices[item.index],
                    ),
                };
            }

            return {
                vector: Array.from(
                    {length: segments.length},
                    (item, index) => index,
                ),
            };
        case SORTING_TYPES.ARRIVAL:
        case SORTING_TYPES.DEPARTURE: {
            if (context.when.special === DateSpecialValue.allDays) {
                const sortTimezone =
                    by === SORTING_TYPES.ARRIVAL
                        ? context.to.timezone
                        : context.from.timezone;

                return {
                    vector: buildVector(
                        segments.map(item => ({
                            moment: moment.tz(item[by], MOMENT, sortTimezone),
                            isInterval: item.isInterval,
                        })),
                        {
                            by: SORTERS.TIME,
                            reverse,
                        },
                    ),
                };
            }

            return {
                vector: buildVector(
                    segments.map(item => ({
                        datetime: item[by],
                        isInterval: item.isInterval,
                    })),
                    {
                        by: SORTERS.DATETIME,
                        reverse,
                    },
                ),
            };
        }

        default:
            return {
                vector: buildVector(segments, sort),
            };
    }
}
