import StationTime from '../../interfaces/state/station/StationTime';
import AllThreadType from '../../interfaces/state/station/AllThreadType';
import StationCompaniesById from '../../interfaces/state/station/StationCompaniesById';
import StationEventList from '../../interfaces/state/station/StationEventList';
import DateRobot from '../../interfaces/date/DateRobot';
import DateMoment from '../../interfaces/date/DateMoment';

import isPlaneThread from './isPlaneThread';
import isRailroadThread from './isRailroadThread';
import getSubtitle from './getSubtitle';
import getCompany from './getCompany';
import getPlaneDirection from './getPlaneDirection';
import getDateRobotFromDateMoment from '../date/getDateRobotFromDateMoment';

interface IGetThread<T extends AllThreadType> {
    event: StationEventList;
    threads: T[];
    search: string;
    isMobile: boolean;
    terminalName: string;
    companiesById: StationCompaniesById;
    time: StationTime | null;
    whenDate?: DateRobot;
}

interface IIsMatchForSearch {
    event: StationEventList;
    thread: AllThreadType;
    isMobile: boolean;
    search: string;
    companiesById: StationCompaniesById;
}

function isMatchForSearch({
    event,
    thread,
    isMobile,
    search,
    companiesById,
}: IIsMatchForSearch): boolean {
    if (isMobile || !search) {
        return true;
    }

    const codeshares =
        'codeshares' in thread && thread.codeshares ? thread.codeshares : null;
    const routeStations =
        'routeStations' in thread ? thread.routeStations : null;

    let text = '';

    if (isRailroadThread(thread)) {
        const {
            transportType,
            isAeroExpress,
            transportSubtype,
            deluxeTrainTitle,
            title,
            number,
        } = thread;

        const subtitle = getSubtitle(
            transportType,
            isAeroExpress,
            transportSubtype,
            deluxeTrainTitle,
        );

        text = `${title} ${number} ${subtitle ?? ''}`;
    }

    if (isPlaneThread(thread)) {
        const {number, companyId} = thread;

        const companyNames = [
            companyId,
            ...(codeshares
                ? codeshares.map(codeshare => codeshare.companyId)
                : []),
        ]
            .map(id => getCompany(id, companiesById)?.title)
            .filter(Boolean)
            .join(' ');

        const companyNumbers = [
            number,
            ...(codeshares
                ? codeshares.map(codeshare => codeshare.number)
                : []),
        ].join(' ');

        const direction = routeStations
            ? getPlaneDirection(routeStations, event)
            : '';

        text = `${direction} ${companyNames} ${companyNumbers}`;
    }

    return text.toLowerCase().includes(search.trim().toLowerCase());
}

function isMatchForTerminal(
    thread: AllThreadType,
    terminalName: string,
): boolean {
    if (!isPlaneThread(thread) || !terminalName) {
        return true;
    }

    const {terminalName: threadTerminalName} = thread;

    return terminalName === threadTerminalName;
}

/* Рейсы с вечера вчерашнего дня, фактически прилетевшие сегодня, должны попадать в интервал с 00:00-02:00 */
function isYesterdaysThreadsInCorrectInterval(
    time: StationTime | null,
): boolean {
    return time === StationTime['p0-2'];
}

function isYesterdaysThread(
    whenDate?: DateRobot,
    datetime?: DateMoment,
): boolean {
    return Boolean(
        whenDate &&
            datetime &&
            whenDate !== getDateRobotFromDateMoment(datetime),
    );
}

function isMatchForTime(
    thread: AllThreadType,
    time: StationTime | null,
    whenDate?: DateRobot,
): boolean {
    if (!isPlaneThread(thread)) {
        return true;
    }

    if (!time || time === StationTime.all) {
        return true;
    }

    const {
        eventDt: {time: timeThread, datetime},
    } = thread;

    if (isYesterdaysThread(whenDate, datetime)) {
        return isYesterdaysThreadsInCorrectInterval(time);
    }

    const [minTime, maxTime] = time.split('-');

    return (
        timeThread >= minTime &&
        timeThread <= (maxTime === '00:00' ? '24:00' : maxTime)
    );
}

export default function getFilteredThreads<T extends AllThreadType>({
    event,
    threads,
    search,
    isMobile,
    terminalName,
    companiesById,
    time,
    whenDate,
}: IGetThread<T>): T[] {
    return threads.filter(thread => {
        if (
            !isMatchForSearch({event, thread, isMobile, search, companiesById})
        ) {
            return false;
        }

        if (!isMatchForTerminal(thread, terminalName)) {
            return false;
        }

        if (!isMatchForTime(thread, time, whenDate)) {
            return false;
        }

        return true;
    });
}
