import IStationScheduleThread from '../../interfaces/state/station/IStationScheduleThread';

import getTimeFromDateMoment from '../date/getTimeFromDateMoment';

type ThreadsWithInterval = {
    threads: IStationScheduleThread[];
    interval: string;
};

/*
Для разбиения ниток по колонкам.
Предполагает, что нитки уже отсортированы по времени.
Сначала разбивает по колонкам в один час.
Если передан параметр maxColumn и количество колонок по одному часу превышает данный параметр,
то попытается объединить колонки по два часа (например может разбить следующим образом: 00:00 - 02:00,  10:00 - 12:00)
*/
export default function getScheduleThreadsAsTable(
    threads: IStationScheduleThread[],

    maxColumn?: number,
): ThreadsWithInterval[] {
    const table = new Map<number, ThreadsWithInterval>();

    // Разбиваем нитки по колонкам в один час
    threads.forEach(thread => {
        const {departureFrom} = thread;
        const time = getTimeFromDateMoment(departureFrom);

        if (!time) {
            return;
        }

        const [hourString] = time.split(':');
        const hour = parseInt(hourString, 10);

        if (!table.has(hour)) {
            table.set(hour, {
                threads: [],
                interval: String(hour),
            });
        }

        const threadsFromTable = table.get(hour);

        if (!threadsFromTable) {
            return;
        }

        threadsFromTable.threads.push(thread);
    });

    if (maxColumn && table.size > maxColumn) {
        // Объединяем четные колонки со следующими за ними нечетными

        table.forEach((hourThreads, hour) => {
            if (hour % 2 !== 0) {
                // Меняем интервал колонки
                hourThreads.interval = `${hour - 1}-${hour}`;

                return;
            }

            const nextOdd = hour + 1;
            const nextOddThreads = table.get(nextOdd);

            // Меняем интервал колонки
            hourThreads.interval = `${hour}-${nextOdd}`;

            if (!nextOddThreads) {
                return;
            }

            hourThreads.threads.push(...nextOddThreads.threads);

            table.delete(nextOdd);
        });
    }

    return Array.from(table.values());
}
