import IBreadcrumb from '../../interfaces/IBreadcrumb';
import StationSubtype from '../../interfaces/state/station/StationSubtype';
import StationEventList from '../../interfaces/state/station/StationEventList';
import StationType from '../../interfaces/state/station/StationType';
import IStationSettlement from '../../interfaces/state/station/IStationSettlement';
import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import {TransportType} from '../transportType';
import CityStations from '../../interfaces/state/station/CityStations';
import ITerminal from '../../interfaces/state/station/ITerminal';
import IStateFlags from '../../interfaces/state/flags/IStateFlags';

import buildTransportUrl from '../url/transportUrl';
import buildTransportCityUrl from '../url/transportCityUrl';
import {stationUrl} from '../url/stationUrl';
import getMainPage from '../url/crumble/getMainPage';
import getCityUrl from '../url/getCityUrl';

import breadcrumbsKeyset from '../../i18n/breadcrumbs';

export function getFirstElement(transportType: TransportType): IBreadcrumb {
    const url = buildTransportUrl({transportType});

    if (!url) {
        return getMainPage('/');
    }

    let transportText;

    switch (transportType) {
        case TransportType.bus:
            transportText = breadcrumbsKeyset('transport-page-title-bus');
            break;
        case TransportType.plane:
            transportText = breadcrumbsKeyset('transport-page-title-plane');
            break;
        case TransportType.train:
            transportText = breadcrumbsKeyset('transport-page-title-train');
            break;
        case TransportType.suburban:
            transportText = breadcrumbsKeyset('transport-page-title-suburban');
            break;
    }

    return {
        url,
        name: transportText,
    };
}

const SubtypeToTransportType: Partial<Record<StationSubtype, TransportType>> = {
    [StationSubtype.train]: TransportType.train,
    [StationSubtype.suburban]: TransportType.suburban,
    [StationSubtype.plane]: TransportType.plane,
    [StationSubtype.tablo]: TransportType.train,
};

const TypeToTransportType: Partial<Record<StationType, TransportType>> = {
    [StationType.bus]: TransportType.bus,
};

export default function getStationBreadcrumbs(
    id: number,
    title: string,
    type: StationType,
    tld: Tld,
    language: Lang,
    flags: IStateFlags,

    subtype?: StationSubtype,
    mainSubtype?: StationSubtype,
    settlement?: IStationSettlement,
    event?: StationEventList,
    cityStations?: CityStations,
    terminal?: ITerminal,
): IBreadcrumb[] {
    const transportType =
        TypeToTransportType[type] ||
        (subtype && SubtypeToTransportType[subtype]);
    const isArrival = event === StationEventList.arrival;

    if (!transportType) {
        // Случай когда нет страницы для данного типа транспорта. Например для водного транспорта
        // В этом случае крошки будут выглядеть "Расписание транспорта Санкт-Петербурга" -> "Северная пристань"

        if (!settlement) {
            return [getMainPage('/'), {name: title}];
        }

        const cityBreadcrumbs = [
            {
                url: getCityUrl(settlement.id, tld, language),
                name: breadcrumbsKeyset('schedule-in-settlement', {
                    titleGenitive: settlement.titleGenitive || settlement.title,
                }),
            },
            {
                name: title,
            },
        ];

        return cityBreadcrumbs;
    }

    const breadcrumbs = [getFirstElement(transportType)];
    const stationTypeStations = cityStations ? cityStations[type] : undefined;
    // В cityStations для типа StationType.railroad могут содержаться станции с подтипом StationSubtype.suburban. Их не нужно считать для формирования ссылки на страницу поездов города (город + транспорт), потому что из-за того, что станция поездов (StationSubtype.train) одна, может возникнуть редирект на страницу станции
    const stationTypeCount = stationTypeStations
        ? stationTypeStations.filter(
              stationTypeStation =>
                  !stationTypeStation.mainSubtype ||
                  stationTypeStation.mainSubtype === subtype ||
                  (stationTypeStation.subtypes &&
                      subtype &&
                      stationTypeStation.subtypes.includes(subtype)),
          ).length
        : 0;

    const transportCityUrl =
        settlement &&
        buildTransportCityUrl({
            transportType,
            slug: settlement.slug,
        });

    if (settlement && transportCityUrl && stationTypeCount > 1) {
        breadcrumbs.push({
            name: settlement.title,
            url: transportCityUrl,
        });
    }

    breadcrumbs.push({
        name: title,
        url: stationUrl({
            id,
            type,
            subtype,
            mainSubtype,
            tld,
            language,
            event: isArrival && terminal ? event : undefined,
        }),
    });

    if (terminal) {
        breadcrumbs.push({
            name: breadcrumbsKeyset('airport-terminal', {
                terminal: terminal.name,
            }),
            url: stationUrl({
                id,
                type,
                subtype,
                mainSubtype,
                tld,
                language,
                event: type === StationType.plane ? event : undefined,
                terminalName: terminal.name,
            }),
        });
    }

    if (isArrival && !terminal) {
        breadcrumbs.push({
            name: breadcrumbsKeyset('arrival'),
        });
    }

    return breadcrumbs;
}
