import moment from 'moment';

import {TIME, ROBOT} from '../date/formats';

import StationTime from '../../interfaces/state/station/StationTime';
import DateMoment from '../../interfaces/date/DateMoment';
import AllThreadType from '../../interfaces/state/station/AllThreadType';
import StationEventList from '../../interfaces/state/station/StationEventList';
import StationCompaniesById from '../../interfaces/state/station/StationCompaniesById';
import DateRobot from '../../interfaces/date/DateRobot';

import getFilteredThreads from './getFilteredThreads';

interface IGetSuitableTimePeriod {
    now: DateMoment; // Текущее время на станции в таймзоне станции
    event: StationEventList;
    threads: AllThreadType[];
    search: string;
    terminalName: string;
    isMobile: boolean;
    companiesById: StationCompaniesById;
}

export default function getSuitableTimePeriod({
    now,
    event,
    threads,
    search,
    terminalName,
    isMobile,
    companiesById,
}: IGetSuitableTimePeriod): StationTime {
    if (threads.length < 100) {
        return StationTime.all;
    }

    const nowMoment = moment.parseZone(now);
    const nowTime = nowMoment.format(TIME);
    const timePeriod = Object.values(StationTime).find(value => {
        if (value === StationTime.all) {
            return false;
        }

        const [min, max] = value.split('-');

        if (nowTime >= min && nowTime < (max === '00:00' ? '24:00' : max)) {
            return true;
        }

        return false;
    });

    if (!timePeriod) {
        throw new Error(
            `Не удалось выбрать временной интервал для текущего времени на станции. now: ${now}`,
        );
    }

    if (terminalName) {
        const threadsForDisplayWithoutTerminal = getFilteredThreads({
            threads,
            search,
            isMobile,
            terminalName,
            time: null,
            event,
            companiesById,
        });

        if (threadsForDisplayWithoutTerminal.length < 100) {
            return StationTime.all;
        }
    }

    const threadsForDisplay = getFilteredThreads({
        threads,
        search,
        isMobile,
        terminalName,
        time: timePeriod,
        event,
        companiesById,
        whenDate: nowMoment.format(ROBOT) as DateRobot,
    });

    return threadsForDisplay.length ? timePeriod : StationTime.all;
}
