import commonKeyset from '../i18n/common';

const isMobile = process.env.PLATFORM === 'mobile';

function normalize(str) {
    return str.replace(/^\s*/, '').replace(/\s+/g, ' ').toLowerCase();
}

export const CHAR_NBSP = '\u00a0'; // неразрывный пробел
export const CHAR_THINSP = '\u2009';
export const CHAR_DASH = '-';
export const CHAR_MIDDLE_DASH = '\u2013';
export const CHAR_EM_DASH = '—'; // long dash
export const CHAR_RIGHT_ARROW = '→';
export const CHAR_LIST_MARKER = '•';

export function isWhitespace(str) {
    return str.trim() === '';
}

export function complete(from, to) {
    const trimmed = normalize(from);

    return from + to.substr(trimmed.length);
}

export function startsWith(target, part) {
    part = normalize(part);
    target = target.toLowerCase();

    return part === target.substr(0, part.length);
}

/**
 * Простая функция кодирования спец символов для вывода в HTML аттрибутах
 * Аналог _.escape([string='']) - который делает тоже самое
 * Универсальным решением был бы пакет https://github.com/mathiasbynens/he
 * @param {string} str
 * @return {string}
 */
export function encodeSpecialCharacters(str) {
    return String(str)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#x27;');
}

export function decodeSpecialCharacters(str) {
    return String(str)
        .replace(/&amp;/g, '&')
        .replace(/&lt;/g, '<')
        .replace(/&gt;/g, '>')
        .replace(/&quot;/g, '"')
        .replace(/&#x27;/g, "'");
}

export function encodeTemplate(strings, ...values) {
    return strings
        .map(
            (str, i) =>
                str + (i < values.length ? encodeURIComponent(values[i]) : ''),
        )
        .join('');
}

export function humanList(strings) {
    if (strings.length <= 1) {
        return strings[0];
    }

    const lastIndex = strings.length - 1;

    return commonKeyset('and', {
        first: strings.slice(0, lastIndex).join(', '),
        second: strings[lastIndex],
    });
}

export function clearMarkup(str) {
    return (
        str
            .replace(/<[^>]*>/g, ' ')
            // eslint-disable-next-line
            .replace(/\&nbsp;/g, ' ')
    );
}

export function mix(base = '', mixin = '') {
    return `${base} ${mixin}`.trim();
}

/**
 * Функция удаляет новые строки на мобильных устройствах
 * @param {string} string - исходная строка
 * @param {boolean} localIsMobile - является ли платформа мобильным устройством
 *
 * @return {string} - итоговая строка
 */
export function replaceNewLinesForMobiles(string, localIsMobile = isMobile) {
    return localIsMobile ? string.replace(/\n/g, '') : string;
}
