import React from 'react';
import xamel from 'xamel';

/**
 * private функция парсинга тэгов из танкера в HTML-тэги реакта
 * Вы спросите, почему нельзя было сразу писать HTML-тэги в танкере? Хороший вопрос!...
 * @param {string} tag - строка с тэгом
 * @return {Object}
 */
function resolveTag(tag) {
    tag = tag.toLowerCase();

    switch (tag) {
        case 'link':
            return {
                name: 'a',
                props: {href: '#'},
            };
        case 'block':
            return {
                name: 'div',
            };
        case 'br':
            return {
                name: 'br',
            };
        case 'h1':
            return {
                name: 'h1',
            };

        default:
            return {
                name: 'span',
            };
    }
}

/**
 * private функция создания React компонента из переданной XML-like ноды
 * @param {Object} node - объект XML-like ноды из парсера Xamel
 * @param {(string|number)} key - любой ключ для итерации React элементов
 * @param {Object} data - дополнительные данные для props компонентов
 * @return {Object} React-компонент
 */
function buildElement(node, key, data) {
    if (typeof node === 'string') {
        return React.createElement('span', {key}, node);
    }

    let children;
    const tag = resolveTag(node.name);

    if (node.children && node.children.length) {
        children = node.children.map((child, index) =>
            buildElement(child, index, data),
        );
    }

    let props = tag.props || {};

    if (
        typeof node.attrs['data-id'] !== 'undefined' &&
        typeof data[node.attrs['data-id']] !== 'undefined'
    ) {
        props = {
            ...props,
            ...data[node.attrs['data-id']],
        };
    }

    props.key = key;

    return React.createElement(tag.name, props, children);
}

/**
 * Парсит кейсет на наличие тэгов вида <Link>sometext</Link> и заменяет их компонентами React
 * @param {string} str - строка текста из кейсета
 * @param {Object} data - набор данных для парсинга, где ключ это ID элемента, а значение это объект аттрибутов
 * @param {boolean} [transformLines=false] - трансформировать ли переносы строк из танкера в br
 * @example
 * parse('Пойти <Link data-id="forest">в лес</Link>', {'forest': { className: 'green' }});
 * // return [<span>Пойти</span>, <a href="#" className="green">в лес</a>]
 * @return {Object[]} вернет массив React компонентов
 */
export default function tankerParse(str, data, transformLines) {
    let result;

    if (transformLines === true) {
        str = str.replace(/\n/g, '<br />');
    }

    xamel.parse(`<data>${str}</data>`, {trim: false}, (err, xml) => {
        if (!err && xml.children[0].name === 'data') {
            result = (xml.children[0].children || []).map((child, index) =>
                buildElement(child, index, data),
            );
        }
    });

    return result;
}
