import {ROBOT} from '../date/formats';

import {TransportType} from '../transportType';

import getThreadUrl from '../url/getThreadUrl';
import getMainPage from '../url/crumble/getMainPage';
import {getMaskValue} from '../date/mask';
import {getFirstElement} from '../../lib/station/getStationBreadcrumbs';
import getBreadCrumbFromThreadToSearch from './getBreadCrumbFromThreadToSearch';

function validateStations(stations) {
    return Array.isArray(stations) && stations.length > 1;
}

/**
 * Возвращает данные станции отправления из stations
 * @param {Object} obj
 * @param {Array<Object>} obj.stations - станции из ручки нитки (thread)
 * @return {*}
 */
export const getStationFrom = ({stations}) => {
    if (!validateStations(stations)) {
        return;
    }

    const station = stations.find(({isStationFrom}) => isStationFrom);

    if (station === stations[stations.length - 1]) {
        return;
    }

    return station || stations[0];
};

/**
 * Возвращает данные станции прибытия из stations
 * @param {Object} obj
 * @param {Array<Object>} obj.stations - станции из ручки нитки (thread)
 * @param {Object} obj.stationFrom - объект станции отправления из stations
 * @return {*}
 */
export const getStationTo = ({stations, stationFrom}) => {
    if (!validateStations(stations)) {
        return;
    }

    const lastStation = stations[stations.length - 1];

    stationFrom = stationFrom || getStationFrom({stations});

    if (!stationFrom || stationFrom === lastStation) {
        return;
    }

    let isAfterStationFrom = false;

    const stationTo = stations.find(station => {
        if (!isAfterStationFrom) {
            if (station === stationFrom) {
                isAfterStationFrom = true;
            }

            return false;
        }

        return station.isStationTo;
    });

    return stationTo || lastStation;
};

export const getFuncDataForDayComponent = ({
    tld,
    language,
    stationFromId,
    stationToId,
    datesInCalendar,
    runDays,
    threads = {},
    threadId: id,
    canonicalUid,
    shouldColorCalendar,
    timezone,
    isToCitySearchContext,
}) => {
    const formattedDates = datesInCalendar.map(date => date.format(ROBOT));

    return date => {
        if (!formattedDates.includes(date.format(ROBOT))) {
            return {};
        }

        const dateThreadId = getMaskValue(date, runDays);
        const dateThreadInfo = threads[dateThreadId] || {};

        const link = getThreadUrl({
            tld,
            language,
            departureFromMoment: date,
            stationFromId,
            stationToId,
            threadUid: id,
            canonicalUid,
            departureFromFormat: ROBOT,
            timezone,
            toCity: isToCitySearchContext,
        });

        return {
            link,
            type: shouldColorCalendar && dateThreadInfo.type,
            current: Boolean(dateThreadInfo.current),
        };
    };
};

/**
 * Возвращает массив с хлебными крошками для страницы нитки
 * @param {string} title - title страницы
 * @param {Array<Object>} stations - станции из ручки нитки (thread)
 * @param {Object} [stationFrom] - объект станции отправления из stations
 * @param {string} transportType
 * @param {boolean} isMobile
 * @param {string} tld
 * @param {string} language
 * @return {*[]}
 */
export const getThreadBreadCrumbs = ({
    title,
    stations,
    stationFrom,
    transportType,
    tld,
    language,
}) => {
    stationFrom = stationFrom || getStationFrom({stations});
    const stationTo = getStationTo({stations});

    if (!stationFrom) {
        throw new Error(`Не удалось определить станцию отправления.
        Data: ${JSON.stringify({title, stations})}`);
    }

    if (!stationTo) {
        throw new Error(`Не удалось определить станцию прибытия.
        Data: ${JSON.stringify({title, stations})}`);
    }

    const {slug: fromSlug, title: titleFrom} = stationFrom;
    const {slug: toSlug, title: titleTo} = stationTo;

    return [
        getMainPage('/'),
        transportType !== TransportType.water
            ? getFirstElement(transportType)
            : undefined,
        getBreadCrumbFromThreadToSearch({
            transportType,
            titleFrom,
            titleTo,
            tld,
            language,

            fromSlug,
            toSlug,
        }),
        {
            name: title,
        },
    ].filter(Boolean);
};
