import zipObjectDeep from 'lodash/zipObjectDeep';
import invert from 'lodash/invert';

import {CASES, CASE_NOMINATIVE, NUMBERS, NUMBER_SINGLE} from './grammar';

import IStationGroup from '../interfaces/state/stationsGroup/IStationGroup';
import IStateDirections from '../interfaces/state/directions/IStateDirections';

import cartesianProduct from './cartesianProduct';
import getValueFromEnum from './enum/getValueFromEnum';

import keyset from '../i18n/transport-types';

export const ALL_TYPE = 'all';
export const PLANE_TYPE = 'plane';
export const TRAIN_TYPE = 'train';
export const SUBURBAN_TYPE = 'suburban';
export const BUS_TYPE = 'bus';
export const WATER_TYPE = 'water';

export enum TransportType {
    plane = 'plane',
    train = 'train',
    suburban = 'suburban',
    bus = 'bus',
    water = 'water',
}

export enum FilterTransportType {
    all = 'all',
    plane = 'plane',
    train = 'train',
    suburban = 'suburban',
    bus = 'bus',
    water = 'water',
}

export const TRANSPORT_TYPE_MAP: Record<TransportType, FilterTransportType> = {
    [TransportType.plane]: FilterTransportType.plane,
    [TransportType.train]: FilterTransportType.train,
    [TransportType.suburban]: FilterTransportType.suburban,
    [TransportType.bus]: FilterTransportType.bus,
    [TransportType.water]: FilterTransportType.water,
};

export enum KeysetTransportType {
    all = 'all',
    plane = 'plane',
    train = 'train',
    suburban = 'suburban',
    bus = 'bus',
    water = 'water',
    lastochka = 'lastochka',
}

// типы транспорта для которых используются слаги в поиске на сегодня
export const TRANSPORT_TYPES_WITH_SLUGS_TODAY = [FilterTransportType.suburban];

// представление всех видов транспорта в url
export const ALL_TRANSPORT_TYPE_IN_REQUEST = 'all-transport';

export const TRANSPORT_TYPES_IN_REQUEST_FOR_SEARCH_WITH_SLUGS = [
    ALL_TRANSPORT_TYPE_IN_REQUEST,
    TRAIN_TYPE,
    PLANE_TYPE,
    BUS_TYPE,
    SUBURBAN_TYPE,
    WATER_TYPE,
];

// Соответствие типов трансопрта в url запроса типам транспорта, используемым в проекте
export const TRANSPORT_TYPE_BY_TYPE_IN_REQUEST = {
    [ALL_TRANSPORT_TYPE_IN_REQUEST]: FilterTransportType.all,
    train: FilterTransportType.train,
    plane: FilterTransportType.plane,
    bus: FilterTransportType.bus,
    suburban: FilterTransportType.suburban,
    water: FilterTransportType.water,
};

// Соответствие типов транспорта, используемым в проекте, типам трансопрта в url запроса
export const TRANSPORT_TYPE_IN_REQUEST_BY_TRANSPORT_TYPE = invert(
    TRANSPORT_TYPE_BY_TYPE_IN_REQUEST,
);

export const SEARCH_TRANSPORT_TYPES = [
    TransportType.train,
    TransportType.suburban,
    TransportType.bus,
    TransportType.water,
    TransportType.plane,
];

export const ALL_TRANSPORT_TYPES = [
    FilterTransportType.all,
    FilterTransportType.plane,
    FilterTransportType.train,
    FilterTransportType.suburban,
    FilterTransportType.bus,
    FilterTransportType.water,
];

export const SEARCHABLE_TRANSPORT_TYPES = ALL_TRANSPORT_TYPES.slice(0, -1);

export const FILTERABLE_TRANSPORT_TYPES = ALL_TRANSPORT_TYPES.slice(1);

// типы транспорта со страницей нитки
export const THREAD_TRANSPORT_TYPES = [
    TransportType.train,
    TransportType.suburban,
    TransportType.bus,
    TransportType.water,
    TransportType.plane,
];

export const TRANSPORT_CITY_TRANSPORT_TYPES = [
    TransportType.train,
    TransportType.suburban,
    TransportType.bus,
    TransportType.plane,
];

export const TRANSPORT_PAGE_TRANSPORT_TYPES = [
    TransportType.train,
    TransportType.suburban,
    TransportType.bus,
    TransportType.plane,
];

export const STATIONS_TYPES_ORDER = [
    TransportType.plane,
    TransportType.train,
    TransportType.bus,
    TransportType.water,
    TransportType.suburban,
];

export const STATION_PAGE_TYPES = [TransportType.train, TransportType.suburban];

export const AVAILABLE_TRANSPORT_TYPES_ORDER = [
    TransportType.suburban,
    TransportType.bus,
    TransportType.train,
    TransportType.plane,
    TransportType.water,
];

export const TRANSPORT_TYPES_WITH_DISPLAYED_STATIONS = [
    TransportType.plane,
    TransportType.train,
    TransportType.bus,
    TransportType.water,
];

export const THREAD_SCHEDULE_TYPES = [
    TransportType.suburban,
    TransportType.train,
];
// Типы транспорта для которых используется упрощенная схема представления различных расписаний
export const THREAD_SCHEDULE_TYPES_SIMPLIFIED = [TransportType.train];

export function parseTransportType(str: string): FilterTransportType {
    const parsedStr = getValueFromEnum(str, FilterTransportType);

    if (parsedStr && SEARCHABLE_TRANSPORT_TYPES.includes(parsedStr)) {
        return parsedStr;
    }

    return FilterTransportType.all;
}

function getTransportForm(
    transportType: TransportType,
    number: string,
    nounCase: string,
): string {
    let key = `type-${transportType}`;

    if (number !== NUMBER_SINGLE) {
        key += `-${number}`;
    }

    if (nounCase !== CASE_NOMINATIVE) {
        key += `-${nounCase}`;
    }

    return keyset.get(key);
}

export function buildTransportForms(transportTypes: TransportType[]): object {
    const product = cartesianProduct(transportTypes, NUMBERS, CASES);
    const props = product.map(properties => properties.join('.'));
    const values = product.map(properties =>
        getTransportForm(properties[0], properties[1], properties[2]),
    );

    return zipObjectDeep(props, values);
}

export function getAvailableTransportTypes(
    stations: IStationGroup[],
    directions: IStateDirections,
): TransportType[] {
    const types = {};

    stations.forEach(station => {
        types[station.t_type] = true;
    });

    if (directions.length) {
        types[TransportType.suburban] = true;
    }

    return AVAILABLE_TRANSPORT_TYPES_ORDER.filter(type => types[type]);
}

export function getTransportTypeName(transportType: TransportType): string {
    return keyset(`type-${transportType}`);
}
