import get from 'lodash/get';

import {SPECIAL_VALUES} from '../../date/values';
import {CHAR_EM_DASH, CHAR_NBSP} from '../../stringUtils';

import {FilterTransportType} from '../../transportType';
import IWhen from '../../../interfaces/state/IWhen';
import IPoint from '../../../interfaces/state/searchForm/IPoint';
import IBreadcrumb from '../../../interfaces/IBreadcrumb';
import ISearchContext from '../../../interfaces/state/search/ISearchContext';
import Tld from '../../../interfaces/Tld';
import Lang from '../../../interfaces/Lang';
import IStatePage from '../../../interfaces/state/IStatePage';

import searchUrl, {getCanonicalUrl} from '../searchUrl';
import {getWhenForAllDays} from '../../search/contextUtils';
import isLastochkaPage from '../../search/isLastochkaPage';
import {isStation} from '../../point/pointType';
import getMainPage from '../crumble/getMainPage';

import breadcrumbsKeyset from '../../../i18n/breadcrumbs';
import transportSelectorKeyset from '../../../i18n/transport-selector';

function getPointPage(
    {from = {} as IPoint},
    location: string,
): IBreadcrumb | null {
    const fromType = isStation(from) ? 'station' : 'city';

    return from.key
        ? {
              url: `${location}/${fromType}/${from.key.substr(1)}`,
              name: from.popularTitle || from.title,
              title: breadcrumbsKeyset(`${fromType}-title`, {
                  from: from.popularTitle || from.title,
              }),
          }
        : null;
}

function getTransportCityPage(
    context: ISearchContext,
    originUrl: string,
): IBreadcrumb | null {
    const {from, transportType} = context;
    const citySlug = get(from, 'settlement.slug');
    const cityTitle = get(from, 'settlement.title');

    return citySlug && cityTitle
        ? {
              url: `${originUrl}/${transportType}/${citySlug}`,
              name: cityTitle,
              title: breadcrumbsKeyset('town-title', {from: cityTitle}),
          }
        : null;
}

/**
 * Ссылка на пункты отправления/прибытия после сужения, на все дни,
 * всеми видами транспорта
 * @param {Object} context
 * @param {string} originUrl
 * @param {string} tld
 * @param {string} language
 * @param {string} keyset
 * @return {*}
 */
function getAllDaysPage(
    context: ISearchContext,
    originUrl: string,
    tld: Tld,
    language: Lang,
): IBreadcrumb | null {
    const {
        from = {} as IPoint,
        to = {} as IPoint,
        language: languageFromContext,
    } = context;

    if (!from.slug || !to.slug) {
        return null;
    }

    const titleFrom = from.popularTitle || from.title;
    const titleTo = to.popularTitle || to.title;

    return {
        url: searchUrl(
            {
                context: {
                    ...context,
                    originalFrom: undefined,
                    originalTo: undefined,
                    transportType: FilterTransportType.all,
                    when: getWhenForAllDays(languageFromContext),
                },
            },
            tld,
            language,
            originUrl,
        ),
        name: breadcrumbsKeyset('on-all-days', {
            points: `${titleFrom}${CHAR_NBSP}${CHAR_EM_DASH} ${titleTo}`,
        }),
        title: breadcrumbsKeyset('on-all-days-title', {
            points: `${titleFrom}${CHAR_NBSP}${CHAR_EM_DASH} ${titleTo}`,
        }),
    };
}

/**
 * Ссылка на пункты отправления/прибытия после сужения, на все дни,
 * данным видом транспорта
 * @param {Object} context
 * @param {string} tld
 * @param {string} language
 * @param {string} originUrl
 * @return {*}
 */
function getAllDaysTransportPage(
    context: ISearchContext,
    tld: Tld,
    language: Lang,
    originUrl: string,
): IBreadcrumb | null {
    const {
        from = {} as IPoint,
        to = {} as IPoint,
        transportType,
        language: languageFromContext,
    } = context;

    return from.slug && to.slug && transportType !== FilterTransportType.all
        ? {
              url: searchUrl(
                  {
                      context: {
                          ...context,
                          originalFrom: undefined,
                          originalTo: undefined,
                          when: getWhenForAllDays(languageFromContext),
                      },
                  },
                  tld,
                  language,
                  originUrl,
              ),
              name: breadcrumbsKeyset(transportType),
              title: transportSelectorKeyset(`title-${transportType}`, {
                  from: from.title,
                  to: to.title,
              }),
          }
        : null;
}

function getDatePage(
    {when = {} as IWhen, language},
    url: string,
): (IBreadcrumb & {dataNoSnippet}) | null {
    if (!when.text) {
        return null;
    }

    return when.text !== SPECIAL_VALUES[language]['all-days']
        ? {
              url,
              name: breadcrumbsKeyset('on-date', {date: when.text}),
              dataNoSnippet: !when.special,
          }
        : null;
}

export default function getCrumbs(
    context,
    tld,
    language,
    page,
    filtering,
): (IBreadcrumb | null)[] {
    const {originUrl, fullUrl} = page;
    const {transportType, when} = context;

    if (isLastochkaPage(transportType, when, filtering)) {
        return getLastochkaCrumbs(context, page, tld, language);
    }

    if (transportType === FilterTransportType.suburban) {
        return getSuburbanCrumbs(context, page);
    }

    return [
        getMainPage(originUrl),
        getPointPage(context, originUrl),
        getAllDaysPage(context, originUrl, tld, language),
        getAllDaysTransportPage(context, tld, language, originUrl),
        getDatePage(context, fullUrl),
    ].filter(Boolean);
}

function getLastochkaCrumbs(
    context: ISearchContext,
    page: IStatePage,
    tld: Tld,
    language: Lang,
): (IBreadcrumb | null)[] {
    const {originUrl, fullUrl} = page;
    const {from, to} = context;

    const titleFrom = from.popularTitle || from.title;
    const titleTo = to.popularTitle || to.title;

    return [
        getMainPage(originUrl),
        getTransportCityPage(context, originUrl),
        isStation(from) ? getPointPage(context, originUrl) : null,
        {
            url: getCanonicalUrl(context, tld, language, originUrl),
            name: breadcrumbsKeyset('current-page-suburban', {
                points: `${titleFrom}${CHAR_NBSP}${CHAR_EM_DASH} ${titleTo}`,
            }),
            title: breadcrumbsKeyset('current-page-suburban-title', {
                points: `${titleFrom}${CHAR_NBSP}${CHAR_EM_DASH} ${titleTo}`,
            }),
        },
        {
            url: fullUrl,
            name: breadcrumbsKeyset('lastochka-page'),
        },
    ].filter(Boolean);
}

function getSuburbanCrumbs(
    context: ISearchContext,
    page: IStatePage,
): (IBreadcrumb | null)[] {
    const {originUrl, fullUrl} = page;
    const {from, to} = context;

    const titleFrom = from.popularTitle || from.title;
    const titleTo = to.popularTitle || to.title;

    return [
        getMainPage(originUrl),
        getTransportCityPage(context, originUrl),
        isStation(from) ? getPointPage(context, originUrl) : null,
        {
            url: fullUrl,
            name: breadcrumbsKeyset('current-page-suburban', {
                points: `${titleFrom}${CHAR_NBSP}${CHAR_EM_DASH} ${titleTo}`,
            }),
            title: breadcrumbsKeyset('current-page-suburban-title', {
                points: `${titleFrom}${CHAR_NBSP}${CHAR_EM_DASH} ${titleTo}`,
            }),
        },
    ].filter(Boolean);
}
