import {SPECIAL_VALUES} from '../date/values';

import {FilterTransportType} from '../transportType';
import Point from '../../interfaces/Point';
import Slug from '../../interfaces/Slug';
import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import DateRobot from '../../interfaces/date/DateRobot';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import ISearchFiltering from '../../interfaces/state/search/ISearchFiltering';

import searchUrl from './searchUrl';
import isDateRobot from '../date/isDateRobot';
import getValueFromEnum from '../enum/getValueFromEnum';

type Original = {
    key?: Point;
    slug?: Slug;
};

interface IGetSearchUrlCommon {
    transportType: FilterTransportType;
    tld: Tld;
    language: Lang;

    when?: DateRobot | DateSpecialValue;
    filtering?: ISearchFiltering;
}

export interface IGetSearchUrlByPoint extends IGetSearchUrlCommon {
    fromPoint: Point;
    toPoint: Point;

    fromSlug?: Slug;
    toSlug?: Slug;
}

interface IGetSearchUrlBySlugForAllDays extends IGetSearchUrlCommon {
    fromSlug: Slug;
    toSlug: Slug;
    when: DateSpecialValue.allDays;
}

interface IGetSearchUrlBySlugForSuburban extends IGetSearchUrlCommon {
    transportType: FilterTransportType.suburban;
    fromSlug: Slug;
    toSlug: Slug;
    when: DateSpecialValue.today;
}

/*
Функция создана как обертка над тем неудачным решением, которое используется в searchUrl,
что предполагает наличие поискового контекста. Но, к сожалению, поисковый контекст есть только на
странице поиска, а нам нужно строить ссылки на поиск и с других страниц.

Данное решение не полное и в будущем потребует расширения.
*/
export default function getSearchUrl(
    params:
        | IGetSearchUrlByPoint
        | IGetSearchUrlBySlugForAllDays
        | IGetSearchUrlBySlugForSuburban,
): string {
    const {transportType, filtering, when, tld, language} = params;

    const originalFrom: Original = {
        key: 'fromPoint' in params ? params.fromPoint : undefined,
        slug: 'fromSlug' in params ? params.fromSlug : undefined,
    };
    const originalTo: Original = {
        key: 'toPoint' in params ? params.toPoint : undefined,
        slug: 'toSlug' in params ? params.toSlug : undefined,
    };

    const dateRobot = when && isDateRobot(when) ? when : undefined;
    const dateSpecial = when && getValueFromEnum(when, DateSpecialValue);
    const dateText =
        dateSpecial && SPECIAL_VALUES[language][dateSpecial]
            ? SPECIAL_VALUES[language][dateSpecial]
            : dateRobot;

    const fakeSearchState = {
        context: {
            transportType,
            originalFrom,
            originalTo,
            when: {
                text: dateText,
                special: dateSpecial,
                date: dateRobot,
            },
        },
        filtering,
    };

    return searchUrl(fakeSearchState, tld, language);
}
