import {Moment} from 'moment';

import {ROBOT_DATETIME} from '../date/formats';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import DateRobot from '../../interfaces/date/DateRobot';

import makeUrl from './makeUrl';

interface IGetThreadUrlBase {
    tld: Tld;
    language: Lang;

    toCity?: boolean;
    stationToId?: number;
    stationFromId?: number;
    departure?: string;
    departureFromDateRobot?: DateRobot;
    departureFromMoment?: Moment;
    departureFromFormat?: string;
    originUrl?: string;
    timezone?: string;
}

interface IGetThreadUrlWithUid extends IGetThreadUrlBase {
    threadUid: string;
    canonicalUid?: string;
}

interface IGetThreadUrlWithCanonicalUid extends IGetThreadUrlBase {
    canonicalUid: string;
    threadUid?: string;
}

type AllParams = IGetThreadUrlWithUid | IGetThreadUrlWithCanonicalUid;

export default function getThreadUrl(params: IGetThreadUrlWithUid): string;
export default function getThreadUrl(
    params: IGetThreadUrlWithCanonicalUid,
): string;
export default function getThreadUrl(params: AllParams): string {
    const {
        tld,
        language,
        departure,
        toCity,
        departureFromDateRobot,
        departureFromMoment,
        departureFromFormat = ROBOT_DATETIME,
        stationFromId,
        stationToId,
        originUrl = '',
        timezone = '',
    } = params;

    const threadUid = 'threadUid' in params ? params.threadUid : undefined;
    const canonicalUid =
        'canonicalUid' in params ? params.canonicalUid : undefined;

    const query: Record<string, string | number | undefined> = {
        station_from: stationFromId, // Станция отправления
        station_to: stationToId, // Станция прибытия
    };

    if (toCity) {
        query.to_city = 1; // Поиск автобусных тарифов с расширением до города
    }

    if (query.station_from && departureFromDateRobot) {
        query.departure_from = departureFromDateRobot;
    } else if (
        query.station_from &&
        departureFromMoment &&
        departureFromMoment.valueOf()
    ) {
        query.departure_from = departureFromMoment.format(departureFromFormat); // Дата отправления из station_from
    } else if (departure) {
        query.departure = departure;
    }

    if (timezone) {
        query.timezone = timezone;
    }

    return makeUrl(
        `${originUrl}/thread/${canonicalUid || threadUid}`,
        tld,
        language,
        query,
    );
}
