import moment, {Moment} from '../../../reexports/moment-timezone';

import {TIME, ROBOT} from '../date/formats';

import Tld from '../../interfaces/Tld';
import ISearchContext from '../../interfaces/state/search/ISearchContext';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';
import ISegment from '../../interfaces/segment/ISegment';
import IStateFlags from '../../interfaces/state/flags/IStateFlags';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import IStateSeoQueryParams from '../../interfaces/state/IStateSeoQueryParams';

import getTravelDomain from './getTravelDomain';
import makeExternalUrl from './makeExternalUrl';
import applyUtm from './applyUtm';
import isCppk from '../segments/isCppk';

interface IGetTrainOrderUrl {
    coachType: string; // Класс вагона (плацкарт, купе, сидячий...)
    context: ISearchContext;
    segment: ISegment | ITransferSegment;
    currencies: IStateCurrencies;
    flags: IStateFlags;
    isProduction: boolean;

    tld?: Tld;
    seoQueryParams?: IStateSeoQueryParams;
    clientId?: string;
    utmMedium?: string;
    departure?: Moment; // Дата и время отправления. Если не задано, то время берется из segment
}

// Необходим чтобы формировать правильные ссылки для электричек ЦППК (https://st.yandex-team.ru/RASPFRONT-8075#5f7c288ae62cb716bb85021c)
const PROVIDER = 'P2';

// Возвращает ссылку покупки на поездатый сегмент
export default function getTrainOrderUrl({
    coachType,
    context,
    segment,
    currencies,
    departure,
    tld,
    flags,
    isProduction,
    seoQueryParams,
    clientId,
    utmMedium,
}: IGetTrainOrderUrl): string {
    const {from, to, userInput, transportType} = context;
    const departureMoment =
        departure ||
        segment.departureMoment ||
        moment.tz(segment.departure, segment.stationFrom.timezone);

    const trainPurchaseNumber =
        'trainPurchaseNumbers' in segment
            ? segment.trainPurchaseNumbers?.[0]
            : undefined;

    // Для электричек и поездов ЦППК необходимо добавлять параметр proveder = P2 для построения верных ссылок на покупку
    const changeProvider = isCppk(segment);

    const originalNumber =
        'originalNumber' in segment ? segment.originalNumber : undefined;
    const threadNumber =
        'thread' in segment ? segment.thread?.number : undefined;

    const query: Record<string, string | number | undefined> = {
        coachType,
        __experiment: flags.__experiment ? 1 : 0,
        number: trainPurchaseNumber || originalNumber || threadNumber,
        fromId: from.key,
        toId: to.key,
        fromName: userInput.from.title,
        toName: userInput.to.title,
        when: departureMoment && departureMoment.format(ROBOT),
        time: departureMoment && departureMoment.format(TIME),
        transportType,
        provider: changeProvider ? PROVIDER : undefined,
        citizenship: tld,
    };

    if (
        currencies &&
        currencies.preferredCurrency !== currencies.nationalCurrency
    ) {
        query.currency = currencies.preferredCurrency;
    }

    const domain = getTravelDomain({isProduction});
    const path = '/trains/order/';

    const url = makeExternalUrl(`${domain}${path}`, query);

    return applyUtm(url, seoQueryParams, clientId, utmMedium);
}
