import {DEFAULT_SORT} from '../sort/utils';
import {
    ALL_TRANSPORT_TYPES,
    ALL_TRANSPORT_TYPE_IN_REQUEST,
    TRANSPORT_TYPE_IN_REQUEST_BY_TRANSPORT_TYPE,
    TRANSPORT_TYPES_WITH_SLUGS_TODAY,
} from '../transportType';
import {LASTOCHKA_FILTER_TYPE} from '../filters/lastochka';

// eslint-disable-next-line no-duplicate-imports
import {FilterTransportType} from '../transportType';

import makeUrl from './makeUrl';
import {getFilteringQuery} from '../filters/filtering';
import isAllDaysSearch from '../search/isAllDaysSearch';
import isTodaySearch from '../search/isTodaySearch';
import isLastochkaPage from '../search/isLastochkaPage';
import {getWhenForAllDays, getWhenForToday} from '../search/contextUtils';
import getLastochkaUrl from './getLastochkaUrl';

/**
 * Функция возвращает поисковый url path для поиска не на все дни
 * @param {string} transportType - тип транспорта
 * @param {boolean} searchNext - является ли поиск поиском на ближайшее время
 * @return {string}
 */
export function composeSearchPath(transportType, searchNext) {
    let path = '/search/';

    if (transportType !== FilterTransportType.all) {
        path += `${transportType}/`;
    }

    if (searchNext) {
        path += 'next/';
    }

    return path;
}

/**
 * Возвращает проежуточный часть url'а, отвечающую за тип транспорта для поиска на все дни
 * @param {string} transportType - тип транспорта
 * @return {string} - соответствующий промежуточный url
 */
export function getTransportIntermediateUrlPart(transportType) {
    if (TRANSPORT_TYPE_IN_REQUEST_BY_TRANSPORT_TYPE[transportType]) {
        return `/${TRANSPORT_TYPE_IN_REQUEST_BY_TRANSPORT_TYPE[transportType]}`;
    }

    return `/${ALL_TRANSPORT_TYPE_IN_REQUEST}`;
}

/**
 * @param {Object} context - объект поискового контекста из search
 * @param {Object} sort - сортировка из search
 * @param {Object} filtering - набор фильтров из search
 * @param {Object} currencies - объект валют и выбранной валюты из state
 * @return {{fromId: *, fromName: *, toId: *, toName: *, plan: *}}
 */
export function getSearchQuery({context, sort, filtering, currencies} = {}) {
    const {
        transportType,
        when,
        from: finalFrom,
        originalFrom,
        to: finalTo,
        originalTo,
    } = context;
    // предпочитаем брать значения из originalFrom и originalTo,
    // т.к. именно там находится информация о пунктах до сужения
    const from = originalFrom || finalFrom || {};
    const to = originalTo || finalTo || {};
    const query = {};
    const currentIsLastochkaPage = isLastochkaPage(
        transportType,
        when,
        filtering,
    );

    if (!needUseUrlWithSlug(context)) {
        query.fromName = from.title;
        query.toName = to.title;
        query.fromId = from.key;
        query.toId = to.key;

        if (!context.searchNext && !currentIsLastochkaPage) {
            query.when = context.when.text;
        }
    }

    if (context.transportType === FilterTransportType.suburban) {
        query.plan = context.plan;
    }

    if (
        currencies &&
        currencies.preferredCurrency !== currencies.nationalCurrency
    ) {
        query.currency = currencies.preferredCurrency;
    }

    if (
        sort &&
        (sort.by !== DEFAULT_SORT.by || sort.reverse !== DEFAULT_SORT.reverse)
    ) {
        query.sortBy = `${sort.reverse ? '-' : ''}${sort.by}`;
    }

    if (filtering) {
        const filteringQuery = getFilteringQuery(filtering);

        Object.keys(filteringQuery).forEach(filteringKey => {
            if (
                currentIsLastochkaPage &&
                filteringKey === LASTOCHKA_FILTER_TYPE
            ) {
                return;
            }

            query[filteringKey] = filteringQuery[filteringKey];
        });
    }

    return query;
}

/**
 * Проверяет необходимость использования урла со слагами
 * @param {Object} searchContext
 * @return {boolean}
 */
export function needUseUrlWithSlug(searchContext) {
    const {fromSlug, toSlug} = getSlugs(searchContext);
    const {transportType} = searchContext;
    const needUseSlugs =
        isAllDaysSearch(searchContext) ||
        (TRANSPORT_TYPES_WITH_SLUGS_TODAY.includes(transportType) &&
            isTodaySearch(searchContext));

    return Boolean(needUseSlugs && fromSlug && toSlug);
}

/**
 * Возвращает объект со слагами из поискового контекста
 * @param {Object} searchContext
 * @return {{fromSlug: (*|string), toSlug: (*|string)}}
 */
export function getSlugs(searchContext) {
    const {
        from = {},
        originalFrom = {},
        to = {},
        originalTo = {},
    } = searchContext;

    // В originalFrom и originalTo содержится информация от parseContext до сужения,
    // что лучше отражает, то, что ввел пользователь в форму поиска
    return {
        fromSlug: originalFrom.slug || from.slug,
        toSlug: originalTo.slug || to.slug,
    };
}

/**
 * Функция генерации готовой ссылки на поисковую выдачу в зависимости от переданных параметров
 * @param {Object} state - state.search
 * @param {string} tld
 * @param {string} language
 * @param {string} originUrl - базовый url (https://domain.ru)
 * @return {string}
 */
export default function searchUrl(state, tld, language, originUrl = '') {
    const {context, filtering} = state;
    let path = originUrl || '';
    const {transportType, when, searchNext} = context;
    const {fromSlug, toSlug} = getSlugs(context);

    switch (true) {
        case isLastochkaPage(transportType, when, filtering):
            path += getLastochkaUrl(fromSlug, toSlug);

            break;
        case needUseUrlWithSlug(context): {
            const whenForUrl = isTodaySearch(context) ? '/today' : '';

            path += getTransportIntermediateUrlPart(transportType);
            path += `/${fromSlug}--${toSlug}${whenForUrl}`;

            break;
        }

        default:
            path += composeSearchPath(transportType, searchNext);
    }

    const query = getSearchQuery(state);

    return makeUrl(path, tld, language, query);
}

/**
 * Возвращает каноническую ссылку для страницы поиска.
 * Может вернуть пустую строку. Это значит, что для данной страницы нет канонической ссылки.
 * К примеру такое возможно, когда каноническая страница возвращает 404.
 * @param {Object} context - поисковый контекст (state.search.context)
 * @param {string} tld
 * @param {string} language
 * @param {string} originUrl - базовый url
 * @param {Object} filtering
 * @param {Objest} archivalData
 * @return {string}
 */
export function getCanonicalUrl(
    context,
    tld,
    language,
    originUrl = '',
    filtering = undefined,
    archivalData = null,
) {
    const {canonical: canonicalFromContext, transportType} = context;

    const canonical =
        canonicalFromContext || (archivalData ? archivalData.canonical : null);

    if (!canonical && !archivalData) {
        // значит поиск на все дни вернет 404
        return '';
    }
    // есть информация о возможности добраться между двумя точками с указанием вида транспорта

    // для поиска ласточками
    if (filtering?.filters?.lastochka?.value) {
        return getLastochkaUrl(
            canonical.pointFrom,
            canonical.pointTo,
            originUrl,
        );
    }

    // для поиска на все дни
    const transportTypeInCanonical = ALL_TRANSPORT_TYPES.includes(
        canonical.transportType,
    )
        ? canonical.transportType
        : FilterTransportType.all;
    const transportTypeForLink =
        transportType === FilterTransportType.suburban
            ? transportType
            : transportTypeInCanonical;
    // для электричек канонической страницей является страница с поиском на сегодня
    const whenForLink =
        transportTypeForLink === FilterTransportType.suburban
            ? getWhenForToday(language)
            : getWhenForAllDays(language);

    return searchUrl(
        {
            context: {
                originalFrom: {
                    slug: canonical.pointFrom,
                },
                originalTo: {
                    slug: canonical.pointTo,
                },
                transportType: transportTypeForLink,
                when: whenForLink,
            },
        },
        tld,
        language,
        originUrl,
    );
}
