import moment from 'moment';

import {FILTER_TRANSLATOR, SPAN_TRANSLATOR} from '../station/stationConstants';
import {ROBOT} from '../date/formats';

import StationEventList from '../../interfaces/state/station/StationEventList';
import {TransportType} from '../transportType';
import StationType from '../../interfaces/state/station/StationType';
import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import StationSubtype from '../../interfaces/state/station/StationSubtype';
import StationDateSpecialValue from '../../interfaces/date/StationDateSpecialValue';
import StationTime from '../../interfaces/state/station/StationTime';
import DateRobot from '../../interfaces/date/DateRobot';

import makeUrl from './makeUrl';
import getSortFunction from '../sort/getSortFunction';
import getPossibleSubtypes from '../station/getPossibleSubtypes';

export const regExpTrackingParameters = /^utm_|^from$/;

const ORDER_GET_PARAMETERS = [
    'event',
    'terminal',
    'date',
    'direction',
    'time',
    'search',
    'stop',
    regExpTrackingParameters,
];

export const sortFunc = getSortFunction(ORDER_GET_PARAMETERS);

interface IStationUrl {
    id: number;
    tld: Tld;
    language: Lang;

    type?: StationType;
    subtypes?: StationSubtype[];
    subtype?: StationSubtype;
    mainSubtype?: StationSubtype;
    isMobile?: boolean;
    date?: DateRobot;
    special?: StationDateSpecialValue;
    event?: StationEventList;
    direction?: string;
    terminalName?: string;
    time?: StationTime;
    search?: string;
    stopId?: number;

    // Дополнительные get-параметры для трэкинга (utm-метки, from от колдунщика)
    trackingParameters?: Record<string, string>;

    // Параметры для ссылок старого стека (при построении ссылок из новой части проекта их не используем)
    filter?: string;
    start?: string;
    span?: string;
}

/*
Внимание!
Функция отдает строку с get-параметрами в определенном порядке (по требованию СЕО-специалистов),
так что если вам потребуется добавить get-параметры к этой ссылке, то воспользуйтесь экспортируемой
функцией для сортировки параметров "sortFunc"
*/
export function stationUrl({
    id,
    tld,
    language,

    type,
    subtypes,
    subtype,
    mainSubtype,
    isMobile = false,
    date,
    special,
    event,
    direction,
    terminalName,
    time,
    search,
    stopId,

    trackingParameters = {},

    filter,
    start,
    span,
}: IStationUrl): string {
    const dateForUrl = getDateForUrl({
        date,
        special,
        span,
        start,
        filter,
    });

    subtype = subtype || mainSubtype;

    const eventForUrl = event === StationEventList.departure ? '' : event;

    const params = {
        ...trackingParameters,
        event: eventForUrl,
        date: dateForUrl,
        direction: type === StationType.railroad ? direction : undefined,
        terminal: terminalName,
        time,
        search: isMobile ? undefined : search,
        stop:
            type === StationType.bus || type === StationType.water
                ? stopId
                : undefined,
    };

    const possibleSubtypes = getPossibleSubtypes(type);
    const shouldAddSubtype =
        subtype &&
        subtype !== mainSubtype &&
        (!subtypes || subtypes.includes(subtype)) &&
        possibleSubtypes.includes(subtype) &&
        possibleSubtypes.length > 1;

    return makeUrl(
        `/station/${id}/${shouldAddSubtype ? `${subtype}/` : ''}`,
        tld,
        language,
        params,
        sortFunc,
    );
}

export function stationsListUrl(
    type: TransportType,
    cityId: number,
    tld: Tld,
    language: Lang,
): string {
    return makeUrl(`/stations/${type}/`, tld, language, {
        city: cityId,
    });
}

interface IGetDateForUrl {
    span?: string;
    filter?: string;
    start?: string;
    special?: StationDateSpecialValue;
    date?: string;
}

export function getDateForUrl({
    span,
    filter,
    start,
    special,
    date,
}: IGetDateForUrl): string {
    if (special) {
        return special === StationDateSpecialValue.today ? '' : special;
    }

    if (date) {
        const parsedDate = moment.parseZone(date);

        if (parsedDate.isValid()) {
            return parsedDate.format(ROBOT);
        }
    }

    if (start) {
        const parsedStart = moment.parseZone(start);

        if (parsedStart.isValid()) {
            return parsedStart.format(ROBOT);
        }
    }

    if (
        filter !== undefined &&
        Object.keys(FILTER_TRANSLATOR).includes(filter)
    ) {
        const translatedFilter = FILTER_TRANSLATOR[filter];

        return translatedFilter === StationDateSpecialValue.today
            ? ''
            : translatedFilter;
    }

    if (span && Object.keys(SPAN_TRANSLATOR).includes(span)) {
        const translatedSpan = SPAN_TRANSLATOR[span];

        return translatedSpan === StationDateSpecialValue.today
            ? ''
            : translatedSpan;
    }

    return '';
}
