import {
    SET_SEGMENTS,
    SET_EMPTY_SEARCH_RESULT,
    SET_SEARCH_CONTEXT,
    SET_TEASERS,
    UPSERT_TRANSFER_SEGMENTS,
    SET_PLANS,
    UPDATE_PRICES,
    SET_STATIC_TARIFFS,
    SET_DYNAMIC_TARIFFS,
    SET_PRICES_ARE_CACHED,
    SET_QUERYING,
    SET_TRANSFER_QUERYING,
    SET_BLABLACAR,
    SET_SORTING,
    SET_FILTER_VALUE,
    SET_FILTERING,
    RESET_FILTERS,
    SET_BADGES,
    SET_PLANE_QUERIES,
    SET_SEARCH_POPULAR_DIRECTIONS_LINKS,
    SET_NEED_INITIAL_REQUEST_FOR_TRAIN_TARIFFS,
    SET_TRAIN_PARTNERS,
    SET_SEGMENT_IS_OPENED,
    SET_SEGMENT_CODESHARES_IS_OPENED,
    SET_SEO_META_INFO,
    SET_IS_LASTOCHKA_PAGE,
    SET_ARCHIVAL_DATA,
    SET_CANONICALS,
    SET_BANNER_INFO,
} from '../actions/search';
import {DEFAULT_SORT} from '../lib/sort/utils';

import Lang from '../interfaces/Lang';
import {FilterTransportType} from '../lib/transportType';

import {sortSegments, updateSortingIndices} from '../lib/sort/sortSegments';
import isPriceQuerying from '../lib/segments/priceQuerying';
import {setFilterValue} from '../lib/filters/filtering';
import addTrainPartnersToSegment from '../lib/segments/addTrainPartnersToSegment';
import getSegmentTitle from '../lib/segments/getSegmentTitle';
import isSegment from '../lib/segments/isSegment';
import {addBlablacarUtmFromDistance} from '../lib/addBlablacarUtmFromDistance';
import {parseSpecial} from '../lib/date/parseImpl';

export const defaultState = {
    sort: {...DEFAULT_SORT},
    context: {
        userInput: {
            from: {title: '', key: '', slug: ''},
            to: {title: '', key: '', slug: ''},
        },
        from: {title: '', key: '', slug: ''},
        originalFrom: {title: '', key: '', slug: ''},
        to: {title: '', key: '', slug: ''},
        originalTo: {title: '', key: '', slug: ''},
        searchNext: false,
        when: {
            text: '',
            hint: '',
            formatted: '',
        },
        time: {
            timezone: 'Europe/Moscow',
            now: Date.now(),
        },
        language: Lang.ru,
        searchForPastDate: false,
        sameSuburbanZone: false,
        transportType: FilterTransportType.all,
        transportTypes: [],
        latestDatatime: '',
        canonical: null,
    },
    segments: [],
    vector: [],
    queryingPrices: false,
    pricesAreCached: false,
    querying: {
        plane: false,
        train: false,
        bus: false,
        transferAll: false,
        transferPlane: false,
        transferTrain: false,
        transferBus: false,
        transferSuburban: false,
    },
    needInitialRequestForTrainTariffs: false,
    filtering: {
        filters: {},
        filteredSegmentIndices: [],
    },
    plans: {},
    sortMapping: {vector: []},
    trainPartners: [],
    blablacar: {
        querying: false,
        banned: false,
        allDaysCheckComplete: true,
        allDaysCheckResult: null,
        tariff: null,
        distance: undefined,
    },
    transportTypes: [],
    isSuburbanSearchResult: false,
    isBusSearchResult: false,
    isLastochkaPage: false,
    stats: {
        hasElectronicTicket: false,
        hasDynamicPricing: false,
    },
    seoMetaInfo: {},
    archivalData: null,
    canonicals: null,
    popularDirectionsLinks: [],
    bannerInfo: undefined,
};

function patchSegments({segments, trainPartners}) {
    if (!segments || !Array.isArray(segments)) {
        return segments;
    }

    segments = segments
        .map(segment =>
            addTrainPartnersToSegment({
                segment,
                trainPartners,
            }),
        )
        .map(segment => {
            segment.title = getSegmentTitle(segment);

            return segment;
        });

    return segments;
}

export function search(state = defaultState, {type, payload}) {
    switch (type) {
        case SET_SEARCH_CONTEXT:
            if (payload.when?.special) {
                return {
                    ...state,
                    queryingPrices: false,
                    querying: defaultState.querying,
                    context: {
                        ...payload,
                        when: parseSpecial(
                            payload.when.text,
                            payload.time,
                            payload.language,
                            payload.language,
                            true,
                        ),
                    },
                };
            }

            return {
                ...state,
                queryingPrices: false,
                querying: defaultState.querying,
                context: payload,
            };
        case SET_EMPTY_SEARCH_RESULT:
            return {
                ...state,
                segments: [],
                queryingPrices: false,
                querying: defaultState.querying,
                context: payload,
                sortMapping: {
                    vector: [],
                },
            };
        case SET_SEGMENTS:
        case UPSERT_TRANSFER_SEGMENTS:
        case SET_DYNAMIC_TARIFFS:
        case SET_PLANE_QUERIES:
        case SET_STATIC_TARIFFS: {
            const {segments} = payload;
            const {trainPartners, context} = state;
            const result = {
                ...state,
                ...payload,
            };

            if (!segments) {
                return result;
            }

            return {
                ...result,
                segments: patchSegments({
                    segments,
                    trainPartners,
                    context,
                }),
            };
        }

        case SET_BLABLACAR: {
            const {blablacar} = payload;

            if (!blablacar) {
                return {
                    ...state,
                    blablacar: defaultState.blablacar,
                };
            }

            const {tariff, distance} = blablacar;

            if (!tariff) {
                return {
                    ...state,
                    blablacar,
                };
            }

            const {orderUrl} = tariff;

            return {
                ...state,
                blablacar: {
                    ...blablacar,
                    tariff: {
                        ...tariff,
                        orderUrl: addBlablacarUtmFromDistance({
                            orderUrl,
                            distance,
                        }),
                    },
                },
            };
        }

        case SET_SORTING: {
            const newState = {
                ...state,
                sort: payload,
            };

            return {
                ...newState,
                sortMapping: sortSegments(newState.segments, newState),
            };
        }

        case UPDATE_PRICES: {
            let {segments} = payload;
            const {trainPartners} = state;

            segments = patchSegments({
                segments,
                trainPartners,
            });

            return {
                ...state,
                ...payload,
                segments,
                sortMapping: sortSegments(segments, state),
            };
        }

        case SET_FILTER_VALUE: {
            const filtering = setFilterValue({
                filtering: state.filtering,
                filterType: payload.type,
                value: payload.value,
                segments: state.segments,
            });
            const newState = {
                ...state,
                filtering,
            };

            return {
                ...newState,
                sortMapping: updateSortingIndices(newState),
            };
        }

        case RESET_FILTERS: {
            const newState = {
                ...state,
                filtering: payload,
            };

            return {
                ...newState,
                sortMapping: updateSortingIndices(newState),
            };
        }

        case SET_FILTERING:
            return {
                ...state,
                filtering: payload,
            };
        case SET_PRICES_ARE_CACHED:
            return {
                ...state,
                pricesAreCached: payload,
            };
        case SET_NEED_INITIAL_REQUEST_FOR_TRAIN_TARIFFS:
            return {
                ...state,
                needInitialRequestForTrainTariffs: payload,
            };
        case SET_TRANSFER_QUERYING:
        case SET_QUERYING: {
            const {
                plane = state.querying.plane,
                train = state.querying.train,
                bus = state.querying.bus,
                transferAll = state.querying.transferAll,
                transferPlane = state.querying.transferPlane,
                transferTrain = state.querying.transferTrain,
                transferBus = state.querying.transferBus,
                transferSuburban = state.querying.transferSuburban,
            } = payload;

            const querying = {
                plane,
                train,
                bus,
                transferAll,
                transferPlane,
                transferTrain,
                transferBus,
                transferSuburban,
            };
            const queryingPrices =
                querying.plane ||
                querying.train ||
                querying.bus ||
                querying.transferAll ||
                querying.transferTrain ||
                querying.transferPlane;

            const segments = state.segments.map(item => {
                if (isSegment(item)) {
                    return {
                        ...item,
                        queryingPrices: isPriceQuerying(querying, item),
                    };
                }

                // обработка пересадки
                return {
                    ...item,
                    queryingPrices: isPriceQuerying(querying, item),
                    segments: item.segments.map(transferSegment => ({
                        ...transferSegment,
                        queryingPrices: isPriceQuerying(
                            querying,
                            transferSegment,
                        ),
                    })),
                };
            });

            return {
                ...state,
                segments,
                queryingPrices,
                querying,
                ...(state.queryingPrices && !queryingPrices
                    ? {
                          sortMapping: sortSegments(segments, {
                              ...state,
                              queryingPrices,
                          }),
                      }
                    : null),
            };
        }

        case SET_TEASERS: {
            return {
                ...state,
                teasers: payload,
            };
        }

        case SET_PLANS: {
            return {
                ...state,
                plans: payload,
            };
        }

        case SET_BADGES: {
            return {
                ...state,
                segments: payload,
            };
        }

        case SET_TRAIN_PARTNERS: {
            return {
                ...state,
                trainPartners: Array.isArray(payload)
                    ? payload
                    : defaultState.trainPartners,
            };
        }

        case SET_SEGMENT_IS_OPENED: {
            const {id, isOpened} = payload;

            return {
                ...state,
                segments: state.segments.map(segment => {
                    return segment.id === id
                        ? {
                              ...segment,
                              isOpened,
                          }
                        : segment;
                }),
            };
        }

        case SET_SEGMENT_CODESHARES_IS_OPENED: {
            const {id, codesharesIsOpened} = payload;

            return {
                ...state,
                segments: state.segments.map(segment => {
                    return segment.id === id
                        ? {
                              ...segment,
                              codesharesIsOpened,
                          }
                        : segment;
                }),
            };
        }

        case SET_SEO_META_INFO: {
            return {
                ...state,
                seoMetaInfo: payload,
            };
        }

        case SET_IS_LASTOCHKA_PAGE: {
            return {
                ...state,
                isLastochkaPage: payload,
            };
        }

        case SET_ARCHIVAL_DATA: {
            return {
                ...state,
                archivalData: payload,
            };
        }

        case SET_CANONICALS: {
            return {
                ...state,
                canonicals: payload,
            };
        }

        case SET_SEARCH_POPULAR_DIRECTIONS_LINKS: {
            return {
                ...state,
                popularDirectionsLinks: payload,
            };
        }

        case SET_BANNER_INFO: {
            return {
                ...state,
                bannerInfo: payload,
            };
        }

        default:
            return state;
    }
}
