import {
    SET_TRANSPORT_TYPE,
    SET_TRANSPORT_TYPE_FROM_USER,
    SET_FROM,
    SET_ORIGINAL_FROM,
    SET_TO,
    SET_ORIGINAL_TO,
    SET_WHEN,
    SWAP,
    SET_ERRORS,
    SET_FORM_DATA,
    SET_USER_INPUT,
    SET_FROM_POINT_FROM_USER,
    SET_TO_POINT_FROM_USER,
    START_VALIDATION_PROCESSING,
    FINISH_VALIDATION_PROCESSING,
} from '../actions/searchForm';
import {SET_TLD} from '../actions/tld';
import {SET_NOW} from '../actions/setNow';
import {SET_LANGUAGE} from '../actions/setLanguage';
import {SET_NATIONAL_VERSION} from '../actions/nationalVersion';
import {SET_CLIENT_SETTLEMENT} from '../actions/setClientSettlement';

import {makeReducer} from '../lib/reduxUtils';
import {updateTime} from '../lib/date/update';
import {getToday} from '../lib/date/utils';
import {parseTransportType} from '../lib/transportType';
import {ERROR_TYPES} from '../lib/errors/errorTypes';
import {prepareAmbiguousErrors} from '../lib/errors/errorUtils';

const DEFAULT_TIMEZONES = {
    ru: 'Europe/Moscow',
    ua: 'Europe/Kiev',
    kz: 'Asia/Almaty',
    by: 'Europe/Minsk',
    uz: 'Asia/Tashkent',
};

function filterErrors(errors, field) {
    return errors.filter(
        error =>
            error.type !== ERROR_TYPES.NO_ROUTES &&
            error.fields &&
            error.fields.indexOf(field) === -1,
    );
}

function samePoints(point1, point2) {
    return (
        point1.key === point2.key &&
        point1.title === point2.title &&
        point1.slug === point2.slug &&
        point1.timezone === point2.timezone
    );
}

function sameDay(time1, time2) {
    time1 = getToday(time1);
    time2 = getToday(time2);

    return time1.isSame(time2, 'date');
}

function setTime(state, time) {
    if (state.time.now === time.now && state.time.timezone === time.timezone) {
        return state;
    }

    if (sameDay(time, state.time)) {
        return {
            ...state,
            time,
        };
    }

    return {
        ...state,
        time,
        when: updateTime(state.when, time),
    };
}

function set(state, field, value) {
    return {
        ...state,
        [field]: value,
        errors: filterErrors(state.errors, field),
    };
}

function setLanguage(state, language) {
    return {...state, language};
}

function setNationalVersion(state, nationalVersion) {
    return {
        ...state,
        defaultTimezone:
            DEFAULT_TIMEZONES[nationalVersion] || DEFAULT_TIMEZONES.ru,
    };
}

function setTransportType(state, transportType) {
    transportType = parseTransportType(transportType);

    return set(state, 'transportType', transportType);
}

/**
 * Должен вызываться при смене типа траспорта через интерфейс пользователем.
 * Будет сброшена информация о точках прибытия/отправления после сужения/расширения,
 * т.к. они могут уже не соответствовать текущему типа транспорта
 * @param {Object} state
 * @param {string} transportType
 * @return {Object}
 */
function setTransportTypeFromUser(state, transportType) {
    const {transportType: currentTransportType} = state;
    let newState = setTransportType(state, transportType);

    if (currentTransportType !== transportType) {
        const {from, to} = newState;

        newState = setFrom(newState, from.title);
        newState = setTo(newState, to.title);
    }

    return newState;
}

function createPoint(point) {
    if (!point) {
        return {title: '', key: '', slug: ''};
    }

    if (typeof point === 'string') {
        return {title: point, key: '', slug: ''};
    }

    return {
        ...point,
        title: point.title || '',
        key: point.key || '',
        slug: point.slug || '',
    };
}

function setFrom(state, from) {
    from = createPoint(from);

    if (samePoints(from, state.from)) {
        return state;
    }

    return setTime(set(state, 'from', from), {
        now: state.time.now,
        timezone: from.timezone || state.from.timezone || state.defaultTimezone,
    });
}

function setOriginalFrom(state, originalFrom) {
    originalFrom = createPoint(originalFrom);

    if (samePoints(originalFrom, state.originalFrom)) {
        return state;
    }

    return set(state, 'originalFrom', originalFrom);
}

function setTo(state, to) {
    to = createPoint(to);

    if (samePoints(to, state.to)) {
        return state;
    }

    return set(state, 'to', to);
}

function setOriginalTo(state, originalTo) {
    originalTo = createPoint(originalTo);

    if (samePoints(originalTo, state.originalTo)) {
        return state;
    }

    return set(state, 'originalTo', originalTo);
}

function swap(state) {
    let newState = {
        ...state,
        userInput: {
            from: state.userInput.to,
            to: state.userInput.from,
        },
    };

    newState = setFrom(setTo(newState, state.from), state.to);
    newState = setOriginalFrom(
        setOriginalTo(newState, state.originalFrom),
        state.originalTo,
    );

    return newState;
}

function setWhen(state, when) {
    return {
        ...set(state, 'when', when),
        searchNext: false,
    };
}

function setNow(state, now) {
    return setTime(state, {
        now,
        timezone: state.time.timezone,
    });
}

function parseErrors(errors) {
    return errors.reduce(
        (errorObject, error) => {
            switch (error.type) {
                case ERROR_TYPES.AMBIGUOUS:
                    return {
                        ...errorObject,
                        [ERROR_TYPES.AMBIGUOUS]: {
                            ...errorObject[ERROR_TYPES.AMBIGUOUS],
                            [[...error.fields]]: prepareAmbiguousErrors(
                                error.variants,
                            ),
                        },
                    };
                case ERROR_TYPES.NO_ROUTES:
                    return errorObject;

                default:
                    return {
                        ...errorObject,
                        errors: [...errorObject.errors, error],
                    };
            }
        },
        {errors: []},
    );
}

function setPlan(state, plan) {
    return {
        ...state,
        plan,
    };
}

function setErrors(state, errors) {
    return {
        ...state,
        ...parseErrors(errors || []),
    };
}

function setFormData(
    state,
    {
        transportType,
        from,
        originalFrom,
        to,
        originalTo,
        when,
        searchNext,
        plan,
        sameSuburbanZone = false,
        errors,
        distance = null,
    },
) {
    state = setTransportType(state, transportType);
    state = setTo(state, to); // информация о точке прибытия после сужения
    state = setOriginalTo(state, originalTo); // информация о точке прибытия до сужения
    state = setWhen(state, when);
    state = setFrom(state, from); // информация о точке оправления после сужения
    state = setOriginalFrom(state, originalFrom); // информация о точке оправления до сужения
    state = setErrors(state, errors);
    state = {...state, searchNext, sameSuburbanZone, distance};
    state = setPlan(state, plan);

    return state;
}

function setFromPointFromUser(state, from) {
    // сбрасываем данные во from,
    // чтобы при смене данных пользователем произошел запрос к ручке parseContext, перед ручкой search
    state = setFrom(state, createPoint(from).title);
    state = setOriginalFrom(state, from);
    state = setUserInput(state, {from});

    return state;
}

function setToPointFromUser(state, to) {
    // сбрасываем данные в to,
    // чтобы при смене данных пользователем произошел запрос к ручке parseContext, перед ручкой search
    state = setTo(state, createPoint(to).title);
    state = setOriginalTo(state, to);
    state = setUserInput(state, {to});

    return state;
}

function setUserInput(state, {from, to}) {
    return {
        ...state,
        userInput: {
            from:
                typeof from !== 'undefined'
                    ? createPoint(from)
                    : state.userInput.from,
            to:
                typeof to !== 'undefined'
                    ? createPoint(to)
                    : state.userInput.to,
        },
    };
}

function setClientSettlement(state, {timezone}) {
    return setTime(state, {
        now: state.time.now,
        timezone,
    });
}

function setTld(state, tld) {
    return {
        ...state,
        tld,
    };
}

function startValidationProcessing(state, payload) {
    return {
        ...state,
        validation: payload,
    };
}

function finishValidationProcessing(state) {
    return {
        ...state,
        validation: null,
    };
}

const reducers = {
    [SET_LANGUAGE]: setLanguage,
    [SET_NATIONAL_VERSION]: setNationalVersion,
    [SET_TRANSPORT_TYPE]: setTransportType,
    [SET_TRANSPORT_TYPE_FROM_USER]: setTransportTypeFromUser,
    [SET_FROM]: setFrom,
    [SET_ORIGINAL_FROM]: setOriginalFrom,
    [SET_TO]: setTo,
    [SET_ORIGINAL_TO]: setOriginalTo,
    [SWAP]: swap,
    [SET_WHEN]: setWhen,
    [SET_TLD]: setTld,
    [SET_NOW]: setNow,
    [SET_ERRORS]: setErrors,
    [SET_FORM_DATA]: setFormData,
    [SET_USER_INPUT]: setUserInput,
    [SET_CLIENT_SETTLEMENT]: setClientSettlement,
    [SET_FROM_POINT_FROM_USER]: setFromPointFromUser,
    [SET_TO_POINT_FROM_USER]: setToPointFromUser,
    [START_VALIDATION_PROCESSING]: startValidationProcessing,
    [FINISH_VALIDATION_PROCESSING]: finishValidationProcessing,
};

const defaultState = {
    transportType: 'all',
    from: {title: '', key: '', slug: ''},
    originalFrom: {title: '', key: '', slug: ''},
    to: {title: '', key: '', slug: ''},
    originalTo: {title: '', key: '', slug: ''},
    when: {text: ''},
    searchNext: false,
    errors: [],
    time: {
        timezone: 'Europe/Moscow',
        now: Date.now(),
    },
    userInput: {
        from: {title: '', key: ''},
        to: {title: '', key: ''},
    },
    defaultTimezone: 'Europe/Moscow',
    language: 'ru',
    tld: 'ru',
    distance: null,
    sameSuburbanZone: false,
    validation: null,
};

export default makeReducer(defaultState, reducers);
