import {createReducer, ActionType} from 'typesafe-actions';
import moment from 'moment';

import {MOMENT} from '../lib/date/formats';

import StationEventList from '../interfaces/state/station/StationEventList';
import IStateStation from '../interfaces/state/station/IStateStation';
import StationType from '../interfaces/state/station/StationType';
import StationCompaniesById from '../interfaces/state/station/StationCompaniesById';

import isDateMoment from '../lib/date/isDateMoment';

import * as actions from '../actions/station';

type Actions = ActionType<typeof actions>;

const defaultNow = moment().format(MOMENT);

if (!isDateMoment(defaultNow)) {
    throw new Error(`defaultNow ("${defaultNow}") in not DateMoment`);
}

const defaultState: IStateStation = {
    id: 0,
    type: StationType.railroad,
    subtypes: [],
    pointType: '',
    title: '',
    hasPopularTitle: false,
    longitude: 0,
    latitude: 0,
    directions: [],
    directionCode: '',
    event: StationEventList.departure,
    trusted: false,
    threads: [],
    teasers: [],
    companies: [],
    companiesById: {},
    terminals: [],
    terminalName: '',
    search: '',
    now: defaultNow,
    stops: [],
    scheduleBlocks: [],
    stopText: '',
    stop: null,
    stationFilterIsOpened: false,

    cityData: undefined,
    popularDirections: undefined,
    mainSubtype: undefined,
    currentSubtype: undefined,
    subway: undefined,
    address: undefined,
    titleGenitive: undefined,
    fullTitle: undefined,
    fullTitleDative: undefined,
    fullTitleGenitive: undefined,
    whenDate: undefined,
    whenSpecial: undefined,
    wayToAirport: undefined,
    iataCode: undefined,
    dtAfter: undefined,
    dtBefore: undefined,
    time: undefined,

    calendarIsOpened: false,
    goneThreadsAreOpened: false,
    moreDirectionsIsOpened: false,
};

export default createReducer<IStateStation, Actions>(defaultState)
    .handleAction(actions.setDataForFetchingPage, state => ({...state}))
    .handleAction(actions.setDataFromAPI, (state, {payload}) => ({
        ...defaultState,
        ...payload,
        teasers: payload.teasers.map(teaser => ({
            ...teaser,
            isOpened: false,
        })),
        companiesById: payload.companies.reduce<StationCompaniesById>(
            (result, company) => {
                result[company.id] = company;

                return result;
            },
            {},
        ),
        stops: payload.stops.sort((a, b) => {
            if (a.majority < b.majority) {
                return -1;
            }

            if (a.majority > b.majority) {
                return 1;
            }

            const aTitle = a.title.toLowerCase();
            const bTitle = b.title.toLowerCase();

            if (aTitle > bTitle) {
                return 1;
            }

            if (aTitle < bTitle) {
                return -1;
            }

            return 0;
        }),
    }))
    .handleAction(actions.setStationCityStations, (state, {payload}) => ({
        ...state,
        cityData: payload,
    }))
    .handleAction(actions.setStationPopularDirections, (state, {payload}) => ({
        ...state,
        popularDirections: payload,
    }))
    .handleAction(actions.setCalendarIsOpened, (state, {payload}) => ({
        ...state,
        calendarIsOpened: payload,
    }))
    .handleAction(actions.setTeaserIsOpened, (state, {payload}) => ({
        ...state,
        teasers: state.teasers.map(teaser => ({
            ...teaser,
            isOpened:
                payload.id === teaser.id ? payload.isOpened : teaser.isOpened,
        })),
    }))
    .handleAction(actions.setGoneThreadsAreOpened, (state, {payload}) => ({
        ...state,
        goneThreadsAreOpened: payload,
    }))
    .handleAction(actions.setSearch, (state, {payload}) => ({
        ...state,
        search: payload,
    }))
    .handleAction(actions.setTime, (state, {payload}) => ({
        ...state,
        time: payload,
    }))
    .handleAction(actions.setTerminalName, (state, {payload}) => ({
        ...state,
        terminalName: payload,
    }))
    .handleAction(actions.setMoreDirectionsIsOpened, (state, {payload}) => ({
        ...state,
        moreDirectionsIsOpened: payload,
    }))
    .handleAction(actions.setStopText, (state, {payload}) => ({
        ...state,
        stopText: payload,
    }))
    .handleAction(actions.setStationFilterIsOpened, (state, {payload}) => ({
        ...state,
        stationFilterIsOpened: payload,
    }))
    .handleAction(actions.setStop, (state, {payload}) => ({
        ...state,
        stop: payload
            ? {
                  id: payload.id,
                  title: payload.title,
                  settlement: payload.settlement,
                  threads: payload.threads.reduce((result, thread) => {
                      result[thread] = true;

                      return result;
                  }, {}),
              }
            : null,
    }));
