import groupBy from 'lodash/groupBy';
import moment from 'moment';

import {
    SET_DATA_FROM_API,
    SET_COMPANY_INFO_IS_OPENED,
    SET_DATA_FOR_FETCHING_PAGE,
    SET_CALENDAR_IS_OPENED,
    SET_MAP_IS_OPENED,
    SET_PREV_STATIONS_IS_OPENED,
    SET_NEXT_STATIONS_IS_OPENED,
    SET_MONTH_INDEX,
    SET_MAP_DATA,
    SET_SEGMENT,
    SET_PRICE_QUERYING,
    SET_BLABLACAR,
    SET_BLABLACAR_QUERYING,
    SET_SCHEDULE_BLOCK_IS_OPENED,
    SET_DELUXE_TRAIN,
} from '../actions/thread';
import {BASIC, ASSIGNMENT} from '../lib/constants/threadScheduleTypes';
import {
    NO_CHANGE_WAGON_RELATION,
    OTHER_NO_CHANGE_WAGON_RELATION,
} from '../lib/constants/threadsRelationTypes';

import {makeReducer} from '../lib/reduxUtils';
import updateQuery from '../lib/url/updateQuery';
import getQuery from '../lib/url/getQuery';
import {addBlablacarUtmFromDistance} from '../lib/addBlablacarUtmFromDistance';

import getFactInfo from './thread/getFactInfo';

const defaultState = {
    id: '',
    canonicalUid: '',
    transportType: '',
    stations: [],
    title: '',
    shortTitle: '',
    number: '',
    company: {
        url: '',
        description: '',
        phone: '',
        address: '',
        email: '',
        title: '',
        strange: false,
        hidden: false,
    },
    capitals: [],
    // На страницу нитки попали с поиска до города
    isToCitySearchContext: false,
    // Таймзона для которой рассчитано время прибытия и отправления в stations
    currentTimezone: '',
    // slug столицы относительно которой рассчитаны capitalTimeOffset в stations
    capitalSlug: '',
    fromStationDepartureLocalDt: '',
    runDays: {},
    runDaysText: '',
    companyInfoIsOpened: false,
    departure: '',
    departureFrom: '',
    stationFrom: '',
    stationTo: '',
    calendarIsOpened: false,
    prevStationsIsOpened: false,
    nextStationsIsOpened: false,
    monthIndex: -1,
    mapIsOpened: false,
    mapData: null,
    isIntervalThread: false,
    isSuburbanBus: false,
    isNoChangeWagon: false,
    beginTime: '',
    endTime: '',
    density: '',
    comment: '',
    tariffsKeys: [],
    segment: null,
    blablacar: {
        querying: false,
        banned: false,
        allDaysCheckComplete: true,
        allDaysCheckResult: null,
        tariff: null,
        distance: undefined,
    },
    typesPriceQuerying: false,
    priceQuerying: false,
    threads: {},
    otherTodayThreads: [],
    relatedThreads: {},
    scheduleBlockIsOpened: false,
    deluxeTrain: null,
    /* Признак экспресса по аналогии с ручками search и station. Присутствует в ответе всегда */
    isExpress: false,
    /* Признак аэроэкспресса по аналогии с ручками search и station. Присутствует в ответе всегда */
    isAeroExpress: false,
};

const stationDefaultState = {
    factInfo: null,
    isFuzzy: false,
    isNoStop: false,
    isCombined: false,
    url: '',
    country: {
        id: 0,
        code: '',
        title: '',
    },
    title: '',
    platform: '',
    isTechnicalStop: false,
    timezone: '',
    hidden: false,
    settlement: {
        id: 0,
        title: '',
    },
    id: 0,
    transitionTime: 0,
    arrivalLocalDt: '',
    departureLocalDt: '',
};

const setMapData = (state, payload) => ({
    ...state,
    mapData: payload || null,
});

const setThreadId = (state, payload) => ({
    ...state,
    id: `${payload || ''}`,
});

const setCanonicalUid = (state, payload) => ({
    ...state,
    canonicalUid: `${payload || ''}`,
});

const setCountry = country =>
    country
        ? {
              id: country.id || 0,
              title: country.title || '',
              code: country.code ? country.code.toLowerCase() : '',
          }
        : stationDefaultState.country;

const setSettlement = settlement =>
    settlement
        ? {
              id: settlement.id || 0,
              title: settlement.title || '',
          }
        : stationDefaultState.settlement;

export const setFactInfo = stationState => {
    if (!stationState) {
        return stationDefaultState.factInfo;
    }

    const factInfo = getFactInfo(stationState);

    return factInfo;
};

const setStation = ({
    isFuzzy,
    isNoStop,
    isCombined,
    isStationFrom,
    isStationTo,
    isTechnicalStop,
    arrivalLocalDt,
    departureLocalDt,
    capitalTimeOffset,
    state,
    country,
    title,
    platform,
    hidden,
    settlement,
    id,
    pageType,
    mainSubtype,
    slug,
}) => {
    const difference =
        departureLocalDt && arrivalLocalDt
            ? moment(departureLocalDt) - moment(arrivalLocalDt)
            : 0;

    return {
        factInfo: setFactInfo(state),
        isFuzzy: Boolean(isFuzzy),
        isNoStop: Boolean(isNoStop),
        isCombined: Boolean(isCombined),
        arrivalLocalDt: arrivalLocalDt || '',
        departureLocalDt: departureLocalDt || '',
        capitalTimeOffset: capitalTimeOffset || '',
        isStationFrom: Boolean(isStationFrom),
        isStationTo: Boolean(isStationTo),
        country: setCountry(country),
        title: title || '',
        platform: platform || '',
        isTechnicalStop: Boolean(isTechnicalStop),
        hidden: Boolean(hidden),
        settlement: setSettlement(settlement),
        id: id || 0,
        transitionTime: difference,
        pageType,
        mainSubtype,
        slug,
    };
};

const setStations = (state, payload) => ({
    ...state,
    stations: Array.isArray(payload) ? payload.map(setStation) : [],
});

const setTransportType = (state, payload) => ({
    ...state,
    transportType: payload,
});

const setTitle = (state, payload) => ({
    ...state,
    title: typeof payload === 'string' ? payload : '',
});

const setShortTitle = (state, payload) => ({
    ...state,
    shortTitle: typeof payload === 'string' ? payload : '',
});

const setNumber = (state, payload) => ({
    ...state,
    number: typeof payload === 'string' ? payload : '',
});

const setCompany = (state, payload) => ({
    ...state,
    company: payload
        ? {
              url: payload.url || '',
              description: payload.description || '',
              phone: payload.phone || '',
              address: payload.address || '',
              email: payload.email || '',
              title: payload.title || '',
              strange: Boolean(payload.strange),
              hidden: Boolean(payload.hidden),
          }
        : defaultState.company,
});

const setCapitalSlug = (state, payload) => ({
    ...state,
    capitalSlug: payload || defaultState.capitalSlug,
});

const setRunDays = (state, payload) => ({
    ...state,
    runDays: payload || '',
});

const setRunDaysText = (state, payload) => ({
    ...state,
    runDaysText: payload || '',
});

const setThreads = (state, payload) => {
    Object.keys(payload).forEach(threadKey => {
        if (payload[threadKey].type === ASSIGNMENT) {
            payload[threadKey].type = BASIC;
        }
    });

    return {
        ...state,
        threads: payload || {},
    };
};

const setOtherTodayThread = ({
    uid,
    title,
    startDepartureTime,
    stopArrivalTime,
    departureDt,
}) => {
    if (
        !uid ||
        !title ||
        !startDepartureTime ||
        !stopArrivalTime ||
        !departureDt
    ) {
        return null;
    }

    return {
        uid,
        title,
        startDepartureTime,
        stopArrivalTime,
        departureDt,
    };
};

const setOtherTodayThreads = (state, payload) => ({
    ...state,
    otherTodayThreads: Array.isArray(payload)
        ? payload.map(setOtherTodayThread).filter(Boolean)
        : defaultState.otherTodayThreads,
});

const setRelatedThread = ({
    relationType,
    canonicalUid,
    title,
    isNoChangeWagon,
}) => {
    if (!relationType || !canonicalUid || !title) {
        return null;
    }

    return {
        relationType:
            isNoChangeWagon && relationType === NO_CHANGE_WAGON_RELATION
                ? OTHER_NO_CHANGE_WAGON_RELATION
                : relationType,
        canonicalUid,
        title,
    };
};

const setIsToCitySearchContext = (state, payload) => ({
    ...state,
    isToCitySearchContext: payload,
});

const setRelatedThreads = (state, payload) => {
    if (!Array.isArray(payload)) {
        return {
            ...state,
            relatedThreads: defaultState.relatedThreads,
        };
    }

    const filteredRelatedThreads = payload
        .map(thread =>
            setRelatedThread({
                ...thread,
                isNoChangeWagon: state.isNoChangeWagon,
            }),
        )
        .filter(Boolean);

    return {
        ...state,
        relatedThreads: groupBy(filteredRelatedThreads, 'relationType'),
    };
};

const setCompanyInfoIsOpened = (state, payload) => ({
    ...state,
    companyInfoIsOpened: Boolean(payload),
});

const setDeparture = (state, payload) => ({
    ...state,
    departure: payload || '',
});

const setDepartureFrom = (state, payload) => ({
    ...state,
    departureFrom: payload || '',
});

const setStationFrom = (state, payload) => ({
    ...state,
    stationFrom: payload || '',
});

const setStationTo = (state, payload) => ({
    ...state,
    stationTo: payload || '',
});

const setCalendarIsOpened = (state, payload) => ({
    ...state,
    calendarIsOpened: Boolean(payload),
});

const setMapIsOpened = (state, payload) => ({
    ...state,
    mapIsOpened: Boolean(payload),
});

const setPrevStationsIsOpened = (state, payload) => ({
    ...state,
    prevStationsIsOpened: Boolean(payload),
});

const setNextStationsIsOpened = (state, payload) => ({
    ...state,
    nextStationsIsOpened: Boolean(payload),
});

const setMonthIndex = (state, payload) => {
    let monthIndex = parseInt(payload, 10);

    if (isNaN(monthIndex) || monthIndex < 0) {
        monthIndex = -1;
    }

    return {
        ...state,
        monthIndex,
    };
};

const setIsIntervalThread = (state, payload) => ({
    ...state,
    isIntervalThread: Boolean(payload),
});

const setIsSuburbanBus = (state, payload) => ({
    ...state,
    isSuburbanBus: Boolean(payload),
});

const setIsNoChangeWagon = (state, payload) => ({
    ...state,
    isNoChangeWagon: Boolean(payload),
});

const setBeginTime = (state, payload) => ({
    ...state,
    beginTime: typeof payload === 'string' ? payload : '',
});

const setEndTime = (state, payload) => ({
    ...state,
    endTime: typeof payload === 'string' ? payload : '',
});

const setDensity = (state, payload) => ({
    ...state,
    density: typeof payload === 'string' ? payload : '',
});

const setComment = (state, payload) => ({
    ...state,
    comment: typeof payload === 'string' ? payload : '',
});

const setTariffsKeys = (state, payload) => ({
    ...state,
    tariffsKeys: Array.isArray(payload) ? payload : [],
});

const setSegment = (state, payload) => {
    const segment = payload || null;

    return {
        ...state,
        segment,
    };
};

const setScheduleBlockIsOpened = (state, payload) => ({
    ...state,
    scheduleBlockIsOpened: Boolean(payload),
});

const setDeluxeTrain = (state, payload) => ({
    ...state,
    deluxeTrain: payload,
});

const setIsExpress = (state, payload) => ({
    ...state,
    isExpress: payload,
});

const setIsAeroExpress = (state, payload) => ({
    ...state,
    isAeroExpress: payload,
});

function getBlablacarOrderUrlWithUtm({orderUrl, distance, isMobile}) {
    const {comuto_cmkt = ''} = getQuery(orderUrl);
    const comutoParams =
        typeof comuto_cmkt === 'string' ? comuto_cmkt.split('_') : [];
    const platformTag = isMobile ? 'MOBILE-FOOTER' : 'DESKTOP-FOOTER';

    comutoParams.push(platformTag);

    const threadOrderUrl = updateQuery(orderUrl, {
        utm_campaign: platformTag,
        comuto_cmkt: comutoParams.join('_'),
    });

    return addBlablacarUtmFromDistance({
        distance,
        orderUrl: threadOrderUrl,
    });
}

const setBlablacar = (state, payload) => {
    if (
        !payload ||
        !payload.blablacar ||
        !Object.keys(payload.blablacar).length
    ) {
        return {
            ...state,
            blablacar: defaultState.blablacar,
        };
    }

    const {blablacar, isMobile} = payload;

    return {
        ...state,
        blablacar: {
            querying: Boolean(blablacar.querying),
            banned: Boolean(blablacar.banned),
            allDaysCheckComplete: Boolean(blablacar.allDaysCheckComplete),
            allDaysCheckResult: blablacar.allDaysCheckResult || null,
            distance: blablacar.distance || null,
            tariff: blablacar.tariff
                ? {
                      ...blablacar.tariff,
                      orderUrl: blablacar.tariff.orderUrl
                          ? getBlablacarOrderUrlWithUtm({
                                orderUrl: blablacar.tariff.orderUrl,
                                distance: blablacar.distance,
                                isMobile,
                            })
                          : undefined,
                  }
                : null,
        },
    };
};

// Задаёт, происходит ли опрос цен для блаблакара
const setBlablacarQuerying = (state, payload) => ({
    ...state,
    blablacar: {
        ...state.blablacar,
        querying: Boolean(payload),
    },
});

// Задаёт, происходит ли любой опрос цен
const setPriceQuerying = (state, payload) => ({
    ...state,
    priceQuerying: Boolean(payload),
});

const setCapitalData = ({timeZone, title, titleGenitive, abbrTitle, slug}) => {
    if (!timeZone || !slug || !title) {
        return null;
    }

    return {
        timezone: timeZone,
        title,
        titleGenitive,
        abbrTitle,
        slug,
    };
};

const setCapitals = (state, payload) => ({
    ...state,
    capitals: Array.isArray(payload)
        ? payload.map(setCapitalData).filter(Boolean)
        : defaultState.capitals,
});

const setCurrentTimezone = (state, payload) => ({
    ...state,
    currentTimezone: payload || '',
});

const setFromStationDepartureLocalDt = (state, payload) => ({
    ...state,
    fromStationDepartureLocalDt: String(payload),
});

const setDataFromAPI = (
    state,
    {
        rtstations,
        threads,
        relatedThreads,
        thread: {
            fullTitle,
            title,
            number,
            uid,
            canonicalUid,
            company,
            capitalSlug,
            runDays,
            daysText,
            transportType,
            otherTodayThreads,
            isInterval,
            isSuburbanBus,
            isNoChangeWagon,
            beginTime,
            endTime,
            density,
            comment,
            tariffsKeys,
            capitals,
            fromStationDepartureLocalDt,
            deluxeTrain,
            isExpress,
            isAeroExpress,
        },
    },
) => {
    state = setCapitals(state, capitals);
    state = setTransportType(state, transportType);
    state = setStations(state, rtstations);
    state = setTitle(state, fullTitle);
    state = setShortTitle(state, title);
    state = setNumber(state, number);
    state = setThreadId(state, uid);
    state = setCanonicalUid(state, canonicalUid);
    state = setCompany(state, company);
    state = setCapitalSlug(state, capitalSlug);
    state = setRunDays(state, runDays);
    state = setRunDaysText(state, daysText);
    state = setIsNoChangeWagon(state, isNoChangeWagon);
    state = setThreads(state, threads);
    state = setOtherTodayThreads(state, otherTodayThreads);
    state = setRelatedThreads(state, relatedThreads);
    state = setCompanyInfoIsOpened(state, false);
    state = setIsIntervalThread(state, isInterval);
    state = setIsSuburbanBus(state, isSuburbanBus);
    state = setBeginTime(state, beginTime);
    state = setEndTime(state, endTime);
    state = setDensity(state, density);
    state = setComment(state, comment);
    state = setTariffsKeys(state, tariffsKeys);
    state = setFromStationDepartureLocalDt(state, fromStationDepartureLocalDt);
    state = setDeluxeTrain(state, deluxeTrain);
    state = setIsExpress(state, isExpress);
    state = setIsAeroExpress(state, isAeroExpress);

    return state;
};

const setDataForFetchingPage = (
    state,
    {
        threadId,
        departure,
        departureFrom,
        stationFrom,
        stationTo,
        currentTimezone,
        isToCitySearchContext,
    },
) => {
    state = defaultState;
    state = setIsToCitySearchContext(state, isToCitySearchContext);
    state = setThreadId(state, threadId);
    state = setDeparture(state, departure);
    state = setDepartureFrom(state, departureFrom);
    state = setStationFrom(state, stationFrom);
    state = setStationTo(state, stationTo);
    state = setCurrentTimezone(state, currentTimezone);

    return state;
};

const reducers = {
    [SET_DATA_FROM_API]: setDataFromAPI,
    [SET_MAP_DATA]: setMapData,
    [SET_COMPANY_INFO_IS_OPENED]: setCompanyInfoIsOpened,
    [SET_DATA_FOR_FETCHING_PAGE]: setDataForFetchingPage,
    [SET_CALENDAR_IS_OPENED]: setCalendarIsOpened,
    [SET_MAP_IS_OPENED]: setMapIsOpened,
    [SET_PREV_STATIONS_IS_OPENED]: setPrevStationsIsOpened,
    [SET_NEXT_STATIONS_IS_OPENED]: setNextStationsIsOpened,
    [SET_MONTH_INDEX]: setMonthIndex,
    [SET_SEGMENT]: setSegment,
    [SET_BLABLACAR]: setBlablacar,
    [SET_BLABLACAR_QUERYING]: setBlablacarQuerying,
    [SET_PRICE_QUERYING]: setPriceQuerying,
    [SET_SCHEDULE_BLOCK_IS_OPENED]: setScheduleBlockIsOpened,
    [SET_DELUXE_TRAIN]: setDeluxeTrain,
};

export default makeReducer(defaultState, reducers);
