import IFactInfo from '../../interfaces/state/stateThread/factInfo/IFactInfo';
import FactTypes from '../../interfaces/state/stateThread/factInfo/FactTypes';
import IDirectionFactInfo from '../../interfaces/state/stateThread/factInfo/IDirectionFactInfo';
import DateMoment from '../../interfaces/date/DateMoment';

import threadKeyset from '../../i18n/thread';

enum FactInfoType {
    fact = 'fact',
    fact_interpolated = 'fact_interpolated',
    possible_ok = 'possible_ok',
    possible_delay = 'possible_delay',
    cancelled = 'cancelled',
}

interface IStationStateDirection {
    minutesFrom: number | null;
    minutesTo: number | null;
    type: FactInfoType;
    factTime: DateMoment; // '2021-05-21T07:51:00+03:00
}

export interface IStationState {
    key: string;
    departure?: IStationStateDirection;
    arrival?: IStationStateDirection;
}

const setFactPart = ({
    factTime,
    minutesFrom,
    minutesTo,
    type,
}: IStationStateDirection): IDirectionFactInfo => {
    if (type === FactInfoType.cancelled) {
        return {
            type: FactTypes.CANCELLED,
        };
    }

    if (type === FactInfoType.possible_delay) {
        return {
            type: FactTypes.POSSIBLE,
        };
    }

    if ((minutesFrom && minutesFrom > 0) || (minutesTo && minutesTo > 0)) {
        if (type === FactInfoType.fact) {
            return {
                type: FactTypes.LATE,
                time: factTime,
            };
        }

        return {
            type: FactTypes.LATE,
        };
    }

    return {
        type: FactTypes.OK,
    };
};

const getMessage = (stationState: IStationState): string => {
    if (stationState.departure) {
        const {minutesFrom, minutesTo, type} = stationState.departure;

        if (type === FactInfoType.cancelled) {
            return threadKeyset('cancelled');
        }

        switch (type) {
            case FactInfoType.fact:
                if (minutesFrom === 0) {
                    return threadKeyset('in-time');
                }

                if (minutesFrom && minutesFrom > 0) {
                    return threadKeyset('late', {minutes: minutesFrom});
                }

                if (minutesFrom) {
                    return threadKeyset('advance', {minutes: minutesFrom});
                }

                break;
            case FactInfoType.fact_interpolated:
                if (minutesFrom === 0 && minutesTo === 0) {
                    return threadKeyset('in-time');
                }

                if (minutesTo && minutesTo > 0) {
                    if (minutesFrom === minutesTo) {
                        return threadKeyset('late', {minutes: minutesFrom});
                    }

                    if (minutesFrom) {
                        return threadKeyset('late-from-to', {
                            minutesFrom,
                            minutesTo,
                        });
                    }
                }

                if (minutesFrom && minutesFrom === minutesTo) {
                    return threadKeyset('advance', {minutes: minutesFrom});
                }

                if (minutesFrom && minutesTo) {
                    return threadKeyset('advance-from-to', {
                        minutesFrom,
                        minutesTo,
                    });
                }

                break;
            case FactInfoType.possible_ok:
                return threadKeyset('expected-in-time');
        }
    }

    if (stationState.arrival) {
        const {minutesFrom, minutesTo, type} = stationState.arrival;

        if (type === FactInfoType.cancelled) {
            return threadKeyset('cancelled');
        }

        switch (type) {
            case FactInfoType.fact:
                if (minutesFrom === 0) {
                    return threadKeyset('arrived-in-time');
                }

                if (minutesFrom && minutesFrom > 0) {
                    return threadKeyset('arrived-late', {minutes: minutesFrom});
                }

                if (minutesFrom) {
                    return threadKeyset('arrived-advance', {
                        minutes: minutesFrom,
                    });
                }

                break;
            case FactInfoType.possible_delay:
                if (!minutesTo && !minutesFrom) {
                    return threadKeyset('possible-delay');
                }

                if (!minutesFrom && minutesTo) {
                    return threadKeyset('possible-delay-to', {
                        minutes: minutesTo,
                    });
                }

                if (!minutesTo && minutesFrom) {
                    return threadKeyset('possible-delay-from', {
                        minutes: minutesFrom,
                    });
                }

                if (minutesFrom && minutesTo) {
                    return threadKeyset('possible-delay-from-to', {
                        minutesFrom,
                        minutesTo,
                    });
                }

                break;
            case FactInfoType.possible_ok:
                return threadKeyset('expected-in-time');
        }
    }

    return '';
};

export default function getFactInfo(stationState: IStationState): IFactInfo {
    return {
        message: getMessage(stationState),
        departure: stationState.departure
            ? setFactPart(stationState.departure)
            : null,
        arrival: stationState.arrival
            ? setFactPart(stationState.arrival)
            : null,
    };
}
