import {batch} from 'react-redux';

import searchErrorHandler from './utils/searchErrorHandler';
import scrollWindow from '../../client/lib/scrollWindow';
import {useSuburbanSearchContext} from '../lib/search/contextUtils';

import {
    setFormDataFromRequest,
    validate,
    getSuburbanWidgetSearchContext,
} from '../actions/searchForm';
import {performSearchRequest} from '../actions/search/search';
import {setTeasers, setSeoMetaInfo} from '../actions/search';
import {setSortingFromRequest} from '../actions/search/sorting';
import {setFilteringFromRequest} from '../actions/search/filters';
import {startFetchingPage, finishFetchingPage} from '../actions/page';
import {requestInitialCurrencyRates} from '../actions/currencies';
import {setNow} from '../actions/setNow';
import {requestTrainPartners} from '../actions/search/prices';
import createUrlChecker from '../lib/createUrlChecker';

export const SEARCH = 'search';
export const SEARCH_FORM_PAGE_NAME = 'searchForm';

export default function search({store, req, res, next, logger, ApiError}) {
    const {dispatch} = store;

    scrollWindow(0);

    batch(() => {
        dispatch(setNow(req));
        dispatch(startFetchingPage(SEARCH));
        dispatch(
            setFormDataFromRequest(
                req.query,
                req.params,
                Boolean(req.searchNext),
            ),
        );
        dispatch(setTeasers({}));
        dispatch(setSeoMetaInfo({}));
    });

    const {searchNext} = req;
    const {searchForm} = store.getState();
    const {
        userInput: {from, to},
    } = searchForm;
    const checker = createUrlChecker(store.getState);

    return Promise.all([
        Promise.resolve().then(() => {
            if (useSuburbanSearchContext({from, to, searchNext})) {
                // обработка переходов по урлам вида /search/next для пригородных элеткричек
                return dispatch(getSuburbanWidgetSearchContext(req));
            }

            return dispatch(validate(req));
        }),
        dispatch(requestTrainPartners({searchContext: searchForm, req})),
    ])
        .then(checker)
        .then(() =>
            batch(() =>
                Promise.all([
                    dispatch(performSearchRequest(req, checker)),
                    dispatch(requestInitialCurrencyRates(req)),
                ]),
            ),
        )
        .then(checker)
        .then(() =>
            batch(() => {
                dispatch(setFilteringFromRequest(req.query, req.params));
                dispatch(setSortingFromRequest(req.query));
                dispatch(finishFetchingPage(SEARCH));

                return res.render();
            }),
        )
        .catch(err => searchErrorHandler(err, {store, res, logger, ApiError}))
        .catch(next);
}
