import {batch} from 'react-redux';

import {LASTOCHKA_FILTER_TYPE} from '../lib/filters/lastochka';
import {GONE_FILTER_TYPE} from '../lib/filters/gone';

import searchErrorHandler from './utils/searchErrorHandler';
import scrollWindow from '../../client/lib/scrollWindow';
import getSearchUrl from '../lib/url/getSearchUrl';
import {
    noSegmentsAllowedByFilters,
    setFilterValue as getFilteringWithNewFilterValue,
} from '../lib/filters/filtering';

import {setFormDataFromRequestWithSlugs, validate} from '../actions/searchForm';
import {performSearchRequest} from '../actions/search/search';
import {setFiltering, setTeasers} from '../actions/search';
import {setSortingFromRequest} from '../actions/search/sorting';
import {getFilteringFromRequest} from '../actions/search/filters';
import {setSeoMetaInfoFromBunker} from '../actions/search/seoMetaInfo';
import {startFetchingPage, finishFetchingPage} from '../actions/page';
import {requestInitialCurrencyRates} from '../actions/currencies';
import {setNow} from '../actions/setNow';
import {requestTrainPartners} from '../actions/search/prices';
import fetchSearchPopularDirections from '../actions/search/popularDirections';
import createUrlChecker from '../lib/createUrlChecker';

export const SEARCH = 'search';
export const LASTOCHKA = 'lastochka';

export default function searchWithSlugs({
    store,
    req,
    res,
    next,
    logger,
    ApiError,
}) {
    const {dispatch} = store;

    scrollWindow(0);

    batch(() => {
        dispatch(setNow(req));
        dispatch(startFetchingPage(SEARCH));
        dispatch(setFormDataFromRequestWithSlugs(req.query, req.params));
        dispatch(setTeasers({}));
    });

    const {searchForm, page} = store.getState();
    const {name, fromSlug, toSlug} = req.params;
    const isLastochkaPage = name === LASTOCHKA;
    const checker = createUrlChecker(store.getState);

    return Promise.all(
        batch(() => [
            dispatch(validate(req)),
            dispatch(requestTrainPartners({searchContext: searchForm, req})),
        ]),
    )
        .then(checker)
        .then(() =>
            batch(() =>
                Promise.all([
                    dispatch(setSeoMetaInfoFromBunker(req, page.location.path)),
                    dispatch(fetchSearchPopularDirections(req)),
                    dispatch(performSearchRequest(req, checker)),
                    dispatch(requestInitialCurrencyRates(req)),
                ]),
            ),
        )
        .then(checker)
        .then(() =>
            batch(() => {
                const originalFiltering = getFilteringFromRequest(
                    req.query,
                    req.params,
                    store.getState,
                );
                let filtering = originalFiltering;

                if (isLastochkaPage) {
                    const {search, tld, language} = store.getState();
                    const {segments, context} = search;
                    const {transportType, when} = context;

                    // Если после применения фильтра ласточек не осталось видимых сегментов
                    // пробуем показать ушедшие
                    if (noSegmentsAllowedByFilters(filtering)) {
                        filtering = getFilteringWithNewFilterValue({
                            filtering,
                            filterType: GONE_FILTER_TYPE,
                            value: true,
                            segments,
                        });
                    }

                    // Если после применения всех фильтров всё ещё нет видимых сегментов
                    // делаем редирект на страницу электричек на все дни
                    if (noSegmentsAllowedByFilters(filtering)) {
                        filtering = getFilteringWithNewFilterValue({
                            filtering: originalFiltering,
                            filterType: LASTOCHKA_FILTER_TYPE,
                            value: false,
                            segments,
                        });

                        const params = {
                            tld,
                            language,
                            transportType,
                            when: when.special || when.date,
                            fromSlug,
                            toSlug,
                            filtering,
                        };

                        return res.redirect(302, getSearchUrl(params));
                    }
                }

                dispatch(setFiltering(filtering));
                dispatch(setSortingFromRequest(req.query));
                dispatch(finishFetchingPage(SEARCH));

                return res.render();
            }),
        )
        .catch(err => searchErrorHandler(err, {store, res, logger, ApiError}))
        .catch(next);
}
