import {batch} from 'react-redux';
import isEqual from 'lodash/isEqual';

import {STATION_POPULAR_DIRECTIONS_LIMIT} from '../lib/station/stationConstants';

import StationEventList from '../interfaces/state/station/StationEventList';
import IRouteMiddlewareParams from '../interfaces/router/IRouteMiddlewareParams';
import StationSubtype from '../interfaces/state/station/StationSubtype';
import StationDateSpecialValue from '../interfaces/date/StationDateSpecialValue';
import StationReqParams from '../interfaces/router/StationReqParams';
import StationReqQuery from '../interfaces/router/StationReqQuery';
import IStationFromApi from '../interfaces/state/station/IStationFromApi';
import ICityData from '../interfaces/state/station/ICityData';
import IStationPopularDirectionsFromApi from '../interfaces/state/station/IStationPopularDirectionsFromApi';
import StationType from '../interfaces/state/station/StationType';
import GetParameters from '../interfaces/lib/url/GetParameters';
import StationTime from '../interfaces/state/station/StationTime';
import IStationStopsFromBackend from '../interfaces/state/station/IStationStopsFromBackend';

import {stationUrl, regExpTrackingParameters} from '../lib/url/stationUrl';
import Error404 from '../lib/errors/Error404';
import isDateRobot from '../lib/date/isDateRobot';
import getValueFromEnum from '../lib/enum/getValueFromEnum';
import getQuery from '../lib/url/getQuery';
import getSuitableTimePeriod from '../lib/station/getSuitableTimePeriod';
import scrollWindow from '../../client/lib/scrollWindow';

import {startFetchingPage, finishFetchingPage} from '../actions/page';
import {
    setDataForFetchingPage,
    setDataFromAPI,
    setStationCityStations,
    setStationPopularDirections,
    setTime,
    setTerminalName,
    setSearch,
    setStop,
    setStopText,
} from '../actions/station';

export const STATION_PAGE_NAME = 'station';

export default async function station({
    store,
    req,
    res,
    next,
    api,
}: IRouteMiddlewareParams): Promise<void> {
    try {
        const {getState, dispatch} = store;
        const {language, tld, isTouch, page, flags, nationalVersion} =
            getState();
        const query = (req.query || {}) as GetParameters<StationReqQuery>;

        const {id: stationIdFromParams, subtype: subtypeFromParams} =
            req.params as StationReqParams;

        const {
            event: eventFromQuery,
            date: dateFromQuery,
            direction: directionFromQuery,
            subdir: subdirFromQuery,
            span: spanFromQuery,
            start: startFromQuery,
            filter: filterFromQuery,
            type: subtypeFromQuery,
            terminal: terminalFromQuery,
            time: timeFromQuery,
            search: searchFromQuery,
            stop: stopFromQuery,
        } = query;

        const stationId = parseInt(stationIdFromParams, 10);

        const date =
            typeof dateFromQuery === 'string' ? dateFromQuery : undefined;
        const subdir =
            typeof subdirFromQuery === 'string' ? subdirFromQuery : undefined; // Данный параметр встречается на старом стеке и содержит строку с направлением
        const direction =
            typeof directionFromQuery === 'string'
                ? directionFromQuery
                : undefined;
        const span =
            typeof spanFromQuery === 'string' ? spanFromQuery : undefined;
        const start =
            typeof startFromQuery === 'string' ? startFromQuery : undefined;
        const filter =
            typeof filterFromQuery === 'string' ? filterFromQuery : undefined;
        const search =
            typeof searchFromQuery === 'string' ? searchFromQuery : undefined;
        const stopId =
            typeof stopFromQuery === 'string'
                ? parseInt(stopFromQuery, 10)
                : undefined;

        const event =
            typeof eventFromQuery === 'string'
                ? getValueFromEnum(eventFromQuery, StationEventList)
                : undefined;

        const subtype =
            typeof subtypeFromParams === 'string' && subtypeFromParams
                ? getValueFromEnum(subtypeFromParams, StationSubtype)
                : typeof subtypeFromQuery === 'string'
                ? getValueFromEnum(subtypeFromQuery, StationSubtype)
                : undefined;

        const dateSpecial =
            typeof date === 'string'
                ? getValueFromEnum(date, StationDateSpecialValue)
                : undefined;

        const dateRobot =
            typeof date === 'string' && isDateRobot(date) ? date : undefined;

        const time =
            typeof timeFromQuery === 'string'
                ? getValueFromEnum(timeFromQuery, StationTime)
                : undefined;

        if (page.current !== STATION_PAGE_NAME) {
            scrollWindow(0);
        }

        batch(() => {
            dispatch(startFetchingPage(STATION_PAGE_NAME));
            dispatch(setDataForFetchingPage());
        });

        const [
            stationData,
            stationStops,
            stationCityStations,
            popularDirections,
        ] = await Promise.all<
            IStationFromApi,
            IStationStopsFromBackend,
            ICityData | undefined,
            IStationPopularDirectionsFromApi | undefined
        >([
            api.execStation(
                {
                    stationId,
                    isMobile: isTouch,
                    language,

                    date: dateRobot || dateSpecial,
                    event,
                    subtype,
                    direction,
                    nationalVersion,
                },
                req,
            ),
            api.execStationStops(
                {
                    stationId,
                    language,

                    date: dateRobot || dateSpecial,
                    event,
                    subtype,
                    direction,
                    nationalVersion,
                    returnForTypes: [StationType.bus, StationType.water],
                },
                req,
            ),
            api.execStationCityStations(
                {
                    stationId,
                    language,
                },
                req,
            ),
            subtype !== StationSubtype.tablo
                ? api.execStationPopularDirections2(
                      {
                          stationId,
                          subtype,
                          limit: STATION_POPULAR_DIRECTIONS_LIMIT,
                          language,
                      },
                      req,
                  )
                : undefined,
        ]);

        const {
            type,
            subtypes,
            notEnoughInfo,
            mainSubtype,
            currentSubtype,
            terminals = [],
        } = stationData;

        if (notEnoughInfo) {
            return res.redirect(302, `/info/station/${stationId}`);
        }

        const terminal = terminals.find(
            terminalObj =>
                terminalObj.name === terminalFromQuery ||
                terminalObj.id === Number(terminalFromQuery),
        );
        const terminalName = terminal && terminal.name;

        const trackingParameters = Object.entries(query).reduce<
            Record<string, string>
        >((result, [key, value]) => {
            if (
                regExpTrackingParameters.test(key) &&
                typeof value === 'string' &&
                value
            ) {
                result[key] = value;
            }

            return result;
        }, {});

        const stationUrlParams = {
            id: stationId,
            tld,
            language,
            flags,

            type,
            subtype: currentSubtype,
            mainSubtype,
            isMobile: isTouch,
            date: dateRobot,
            special: dateSpecial,
            event,
            direction: subdir || direction,
            terminalName,
            time,
            search,
            stopId,

            trackingParameters,

            filter,
            start,
            span,
        };

        // кейс, когда запрашивается тип транспорта, которого нет на этой станции
        if (!currentSubtype && (mainSubtype || subtypes.length)) {
            const code = type === StationType.plane ? 301 : 302;

            return res.redirect(
                code,
                stationUrl({
                    ...stationUrlParams,
                    subtype: mainSubtype || subtypes[0],
                }),
            );
        }

        // кейс, когда явно запрашивается страница электрички по новой схеме,
        // при том, что кроме электричек на станции ничего не ходит
        if (
            subtypeFromParams === StationSubtype.suburban &&
            mainSubtype === StationSubtype.suburban
        ) {
            return res.redirect(302, stationUrl(stationUrlParams));
        }

        // кейс, когда параметр time имеет неверное значение
        if (timeFromQuery && timeFromQuery !== time) {
            return res.redirect(302, stationUrl(stationUrlParams));
        }

        // кейс, когда для тачевого урл есть параметр search
        // так как на таче нет поиска по ниткам, делаем редирект
        if (isTouch && search) {
            return res.redirect(302, stationUrl(stationUrlParams));
        }

        // Получаем образцовый урл для данного запроса и если он не совпадает с реальным,
        // за исключением utm-меток, которые идут в конце, делаем редирект
        const {originalUrl} = req;
        const perfectUrl = stationUrl(stationUrlParams);
        const perfectUrlWithoutTrackingParameters = stationUrl({
            ...stationUrlParams,
            trackingParameters: undefined,
        });
        const originalUrlWithouTrackingParameters = originalUrl.replace(
            /[?&](utm_|from).*$/,
            '',
        );
        // убедимся в том, что отрезав последнюю часть урла с utm-метками, не пропали другие параметры
        const originalParametersFilterTrackingParameters = Object.entries(
            getQuery(originalUrl),
        )
            .filter(([key]) => !regExpTrackingParameters.test(key))
            .reduce((result, [key, value]) => {
                result[key] = value;

                return result;
            }, {});
        const originalParametersWithoutTrackingParameters = getQuery(
            originalUrlWithouTrackingParameters,
        );
        const originalDeletedOnlyTrackingParameters = isEqual(
            originalParametersFilterTrackingParameters,
            originalParametersWithoutTrackingParameters,
        );

        if (
            originalDeletedOnlyTrackingParameters
                ? originalUrlWithouTrackingParameters !==
                  perfectUrlWithoutTrackingParameters
                : originalUrl !== perfectUrl
        ) {
            return res.redirect(301, perfectUrl);
        }

        batch(() => {
            dispatch(
                setDataFromAPI({
                    ...stationStops,
                    ...stationData,
                }),
            );
            dispatch(setSearch(!isTouch && search ? search : ''));

            if (terminalName) {
                dispatch(setTerminalName(terminalName));
            }

            if (time) {
                dispatch(setTime(time));
            } else if (type === StationType.plane) {
                // Выбираем период времени для показа ниток
                const {station: stationState} = getState();

                dispatch(
                    setTime(
                        getSuitableTimePeriod({
                            now: stationState.now,
                            threads: stationState.threads,
                            search: stationState.search,
                            terminalName: stationState.terminalName,
                            isMobile: isTouch,
                            event: stationState.event,
                            companiesById: stationState.companiesById,
                        }),
                    ),
                );
            }

            const stop = stationStops.stops.find(({id}) => id === stopId);

            // Кейс, когда в get-параметре stop указана id станции, которой нет
            if (stopId && !stop) {
                return res.redirect(
                    302,
                    stationUrl({
                        ...stationUrlParams,
                        stopId: undefined,
                    }),
                );
            }

            if (stop) {
                dispatch(setStop(stop));
                dispatch(setStopText(stop.title));
            }

            if (stationCityStations) {
                dispatch(setStationCityStations(stationCityStations));
            }

            if (currentSubtype !== StationSubtype.tablo && popularDirections) {
                dispatch(setStationPopularDirections(popularDirections));
            }

            dispatch(finishFetchingPage(STATION_PAGE_NAME));
        });

        return await res.render();
    } catch (error) {
        // Ошибка в апи
        if (error.statusCode === 404) {
            // eslint-disable-next-line no-ex-assign
            error = new Error404();
        }

        // Для 404 ошибок на странице станции нового стека не проксируем их на старый стек
        if (error instanceof Error404 && 'set' in res) {
            res.set('X-Rasp-Prevent-Old-Stack-Proxy', '1');
        }

        return next(error);
    }
}
