import {batch} from 'react-redux';

import {PLANE_TYPE} from '../lib/transportType';

import EnvironmentType from '../interfaces/EnvironmentType';

import ThreadPricePolling from '../lib/polling/ThreadPricePolling';
import ThreadBlablacarPolling from '../lib/polling/ThreadBlablacarPolling';
import setupThreadPricePolling from '../../client/listeners/setupThreadPricePolling';
import setupThreadBlablacarPolling from '../../client/listeners/setupThreadBlablacarPolling';
import isPoint from '../lib/point/isPoint';
import getPointId from '../lib/point/getPointId';
import getPlaneThreadUrl from '../lib/url/getPlaneThreadUrl';
import scrollWindow from '../../client/lib/scrollWindow';
import Error404 from '../lib/errors/Error404';

import {startFetchingPage, finishFetchingPage} from '../actions/page';
import {
    setDataFromAPI,
    setDataForFetchingPage,
    setMapData,
    setPriceQuerying,
    setBlablacarQuerying,
} from '../actions/thread';
import {requestInitialCurrencyRates} from '../actions/currencies';
import {is404ApiError} from '../lib/errors/api/is404ApiError';

export const THREAD_PAGE_NAME = 'thread';

const getIdStation = point => {
    return isPoint(point) ? getPointId(point) : '';
};

export default function thread({store, req, res, next, api, logger, ApiError}) {
    scrollWindow(0);

    const {dispatch, getState} = store;
    const {language, tld, environment, user} = getState();

    const {threadId} = req.params;
    const {
        departure,
        departure_from: departureFrom,
        station_from,
        station_to,
        point_from,
        point_to,
        to_city,
        timezone,
    } = req.query || {};

    const stationFrom = station_from || getIdStation(point_from);
    const stationTo = station_to || getIdStation(point_to);
    const country = tld.toString().toUpperCase();

    try {
        batch(() => {
            dispatch(startFetchingPage(THREAD_PAGE_NAME));
            dispatch(
                setDataForFetchingPage({
                    threadId,
                    departure,
                    departureFrom,
                    stationFrom,
                    stationTo,
                    currentTimezone: timezone,
                    isToCitySearchContext: Boolean(to_city),
                }),
            );
        });
    } catch (err) {
        return next(err);
    }

    const threadParams = {
        threadId,
        country,
        language,
        departure,
        departureFrom,
        stationFrom,
        stationTo,
        timezone,
    };

    return Promise.all([
        api
            .execThread2(
                {
                    isCitySearch: Number(to_city) === 1,
                    ...threadParams,
                },
                req,
            )
            .catch(err => {
                if (err.statusCode === 404) {
                    err = new Error404();
                }

                throw err;
            }),
        api.exec('threadMap', threadParams, req).catch(err => {
            if (err.statusCode !== 404 && !is404ApiError(err, ApiError)) {
                logger.error('routes/thread.js', err, {req});
            }

            return null;
        }),
        dispatch(requestInitialCurrencyRates(req)),
    ])
        .then(([threadData, threadMapData]) =>
            batch(() => {
                const {number: numberPlane, transportType} = threadData.thread;

                if (transportType === PLANE_TYPE) {
                    return res.redirect(
                        301,
                        getPlaneThreadUrl({
                            numberPlane,
                            departure,
                            departureFrom,
                            query: {
                                utm_campaign: 'redirect301',
                                utm_medium: 'flight_landing',
                            },
                        }),
                    );
                }

                dispatch(setDataFromAPI(threadData));
                dispatch(setMapData(threadMapData));

                // если это серверный рендеринг, то
                // производим запрос цен, но не дожидаемся результата,
                // чтобы при запросе с клиента они отдавались быстрее
                if (
                    environment.type === EnvironmentType.server &&
                    !user.isBot
                ) {
                    const threadPricePolling = new ThreadPricePolling({
                        store,
                        api,
                    });
                    const threadBlablacarPolling = new ThreadBlablacarPolling({
                        store,
                        api,
                    });

                    if (threadPricePolling.needRequestPrices()) {
                        dispatch(setPriceQuerying(true));
                        threadPricePolling.requestPrices({req});
                    }

                    if (threadBlablacarPolling.needRequestPrices()) {
                        dispatch(setBlablacarQuerying(true));
                        threadBlablacarPolling.requestPrices({req});
                    }
                }

                if (environment.type === EnvironmentType.client) {
                    setupThreadPricePolling(store, api);
                    setupThreadBlablacarPolling(store, api);
                }

                dispatch(finishFetchingPage(THREAD_PAGE_NAME));

                return res.render();
            }),
        )
        .catch(next);
}
