import {batch} from 'react-redux';

import {getWhenForToday} from '../lib/search/contextUtils';
import scrollWindow from '../../client/lib/scrollWindow';

import {startFetchingPage, finishFetchingPage} from '../actions/page';
import {setDataForFetchingPage, setDataFromAPI} from '../actions/transport';
import {
    setFromPointFromUser,
    setToPointFromUser,
    setTransportType,
    setWhen,
} from '../actions/searchForm';

const REDIRECT_TO_MAIN_PAGE = 'redirectToMainPage';

export const TRANSPORT_PAGE_NAME = 'transport';

export default function transport({store, req, res, next, api}) {
    scrollWindow(0);

    const {dispatch, getState} = store;
    const {language, tld} = getState();
    const {transportType} = req.params;

    batch(() => {
        dispatch(startFetchingPage(TRANSPORT_PAGE_NAME));
        dispatch(setWhen(getWhenForToday(language)));
        dispatch(setFromPointFromUser(''));
        dispatch(setToPointFromUser(''));
        dispatch(setTransportType(transportType));
        dispatch(setDataForFetchingPage({transportType}));
    });

    return Promise.all([
        api
            .exec(
                'transport',
                {
                    language,
                    transportType,
                    countryCode: tld.toUpperCase(),
                },
                req,
            )
            .catch(err => {
                if (err.statusCode === 404) {
                    throw new Error(REDIRECT_TO_MAIN_PAGE);
                }

                throw err;
            })
            .then(transportData => {
                const {
                    mainCity: {stations},
                    cities,
                } = transportData;

                if (stations.length === 0 && (!cities || cities.length === 0)) {
                    throw new Error(REDIRECT_TO_MAIN_PAGE);
                }

                return transportData;
            }),

        api
            .exec(
                'articles',
                {
                    language,
                    country: tld,
                    transportType,
                },
                req,
            )
            .then(response => response.result.articles),
    ])
        .then(([transportData, articles]) =>
            api
                .exec(
                    'transportPopularDirections',
                    {
                        language,
                        slug: transportData.mainCity.slug,
                        transportType,
                        limit: 10,
                    },
                    req,
                )
                .then(result => [
                    transportData,
                    articles,
                    result.fromSettlement,
                ]),
        )
        .then(([transportData, articles, popularDirections]) =>
            batch(() => {
                const {
                    mainCity: {slug, title},
                } = transportData;

                dispatch(setFromPointFromUser({title, slug}));
                dispatch(
                    setDataFromAPI({
                        ...transportData,
                        articles,
                        popularDirections,
                    }),
                );
                dispatch(finishFetchingPage(TRANSPORT_PAGE_NAME));

                return res.render();
            }),
        )
        .catch(err => {
            if (err.message === REDIRECT_TO_MAIN_PAGE) {
                return res.redirect(302, '/');
            }

            throw err;
        })
        .catch(next);
}
