import {batch} from 'react-redux';

import Platform from '../interfaces/Platform';

import Error404 from '../lib/errors/Error404';
import cityUrl from '../lib/url/getCityUrl';
import {stationUrl} from '../lib/url/stationUrl';
import {getWhenForToday} from '../lib/search/contextUtils';
import scrollWindow from '../../client/lib/scrollWindow';
import getSubtypeByTransportType from '../lib/station/getSubtypeByTransportType';

import {finishFetchingPage, startFetchingPage} from '../actions/page';
import {setDataForFetchingPage, setDataFromAPI} from '../actions/transportCity';
import {
    setFromPointFromUser,
    setToPointFromUser,
    setTransportType,
    setWhen,
} from '../actions/searchForm';

export const TRANSPORT_CITY_PAGE_NAME = 'transportCity';

const isMobile = process.env.PLATFORM === Platform.mobile;

export default function transportCity({store, req, res, next, api}) {
    scrollWindow(0);

    const {dispatch, getState} = store;
    const {language, tld} = getState();
    const {transportType, citySlug} = req.params;

    batch(() => {
        dispatch(startFetchingPage(TRANSPORT_CITY_PAGE_NAME));
        dispatch(setFromPointFromUser(''));
        dispatch(setToPointFromUser(''));
        dispatch(setWhen(getWhenForToday(language)));
        dispatch(setTransportType(transportType));
        dispatch(setDataForFetchingPage({transportType}));
    });

    return Promise.all([
        api
            .exec(
                'settlement',
                {
                    language,
                    slug: citySlug,
                },
                req,
            )
            .catch(err => {
                if (err.statusCode === 404) {
                    throw new Error404();
                }

                throw err;
            })
            .then(settlement => {
                if (settlement === null) {
                    throw new Error404();
                }

                return settlement;
            }),

        api
            .exec(
                'stations',
                {
                    settlementId: citySlug,
                    language,
                    transportType,
                },
                req,
            )
            .catch(err => {
                if (err.statusCode === 404) {
                    throw new Error404();
                }

                throw err;
            })
            .then(result => result.connected.concat(result.related)),

        api
            .exec(
                'transportPopularDirections',
                {
                    language,
                    slug: citySlug,
                    transportType,
                    limit: 10,
                },
                req,
            )
            .then(result => result.fromSettlement),

        api
            .exec(
                'articles',
                {
                    language,
                    country: tld,
                    settlementSlug: citySlug,
                    transportType,
                    limit: 2,
                },
                req,
            )
            .then(answer => answer.result.articles),
    ])
        .then(([settlement, stations, popularDirections, articles]) =>
            batch(() => {
                if (!stations || !stations.length) {
                    return res.redirect(302, cityUrl(settlement.id));
                }

                if (stations.length === 1) {
                    const station = stations[0];

                    return res.redirect(
                        302,
                        stationUrl({
                            id: station.id,
                            pageType: station.pageType,
                            mainSubtype: station.mainSubtype,
                            subtype: getSubtypeByTransportType(transportType),
                            isMobile,
                            tld,
                            language,
                        }),
                    );
                }

                dispatch(setFromPointFromUser(settlement));
                dispatch(
                    setDataFromAPI({
                        settlement,
                        stations,
                        popularDirections,
                        articles,
                    }),
                );
                dispatch(finishFetchingPage(TRANSPORT_CITY_PAGE_NAME));

                return res.render();
            }),
        )
        .catch(next);
}
