import {ApiMethod} from 'bla';
import yaConfig from '@yandex-int/yandex-config';

import {Request} from 'express';

import logger from '../logger';
import {apiTimedGot} from '../helpers/timedGot';
import requestIsFailed from './utils/requestIsFailed';

const config = yaConfig();

const defaultAnswer = {
    isDefault: true,
    allDaysTariffs: {
        tariffs: [],
    },
};

module.exports = new ApiMethod({
    name: 'allDaysTariffs',
    params: {
        context: {
            type: 'Object',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        clientSettlementId: {
            type: 'Number',
            required: true,
        },
    },
    action: ({context, nationalVersion, clientSettlementId}, req: Request) => {
        const {transportType, from, to, when, language} = context;
        const query = {
            pointFrom: from.key,
            pointTo: to.key,
            date: [when.date, when.nextDate],
            national_version: nationalVersion,
            clientSettlementId,
            transportType: transportType === 'all' ? undefined : transportType,
        };

        return apiTimedGot(
            {
                ...config.api.options,
                path: `/${language}/segments/min-tariffs/`,
                query,
                json: true,
            },
            req,
            {
                operationName: '/:language/segments/min-tariffs/',
            },
        )
            .then(({body}) => {
                if (requestIsFailed(body)) {
                    throw new Error(
                        `Request is failed: ${JSON.stringify(body.errors)}`,
                    );
                }

                return {allDaysTariffs: body};
            })
            .catch(err => {
                logger.error('server/api/allDaysTariffs', err);

                return defaultAnswer;
            });
    },
});
