import {ApiMethod} from 'bla';
import {Request} from 'express';
import yaConfig from '@yandex-int/yandex-config';

import IExpressRequest from '../../common/interfaces/IExpressRequest';

import isRidesharingEnabled from '../../common/lib/segments/ridesharing/isRidesharingEnabled';

import {apiTimedGot} from '../helpers/timedGot';
import isDirtyBot from '../helpers/isDirtyBot';
import logger from '../logger';
import {getRidesharingProviderByFlags} from './utils/getRidesharingProviderByFlags';

const config = yaConfig();

const BLABLACAR_DEFAULT_DISTANCE = 25;

const defaultAnswer = {
    blablacar: {
        querying: false,
        tariff: null,
        banned: true,
        distance: undefined,
    },
};

module.exports = new ApiMethod({
    name: 'blablacar3',
    params: {
        pointFrom: {
            type: 'String',
            required: true,
        },
        pointTo: {
            type: 'String',
            required: true,
        },
        nationalVersion: {
            type: 'String',
            required: true,
        },
        isMobile: {
            type: 'Boolean',
            required: true,
        },
        date: {
            type: 'String',
        },
        poll: {
            type: 'Boolean',
            defaultValue: false,
        },
        flags: {
            type: 'Object',
            defaultValue: {},
        },
    },
    action: (
        {pointFrom, pointTo, date, nationalVersion, poll, isMobile, flags},
        req: Request & Pick<IExpressRequest, 'isBot' | 'isSearchBot'>,
    ) => {
        if (!isRidesharingEnabled(flags)) {
            return defaultAnswer;
        }

        if (!poll && isDirtyBot(req)) {
            return defaultAnswer;
        }

        const {blablacar: blablacarConfig} = config;
        const provider = getRidesharingProviderByFlags(flags);
        const options = {
            ...blablacarConfig.requestOptions,
            path: `/${poll ? 'poll/' : ''}`,
            query: {
                pointFrom,
                pointTo,
                ...(date ? {date} : null),
                national_version: nationalVersion,
                isTouch: isMobile ? 'true' : 'false',
                minimalDistance: BLABLACAR_DEFAULT_DISTANCE,
                provider,
            },
            json: true,
        };

        return apiTimedGot(options, req, {
            operationName: `/${poll ? 'poll/' : ''}`,
        })
            .then(({body}) =>
                body.errors
                    ? defaultAnswer
                    : {
                          blablacar: {
                              ...body,
                              banned: Boolean(body.banned),
                          },
                      },
            )
            .catch(err => {
                if (err.statusCode !== 428) {
                    logger.error('server/api/blablacar', err);
                }

                return defaultAnswer;
            });
    },
});
